<?php
/**
 * 405 X Gobot.su Dashboard
 * Design from 405 Dashboard
 */

// Suppress errors
error_reporting(E_ALL);
ini_set('display_errors', 1);
ob_start();

// Cookie-based authentication check
$cookieName = 'gs_auth_token';
function dashVerifyAuth() {
    $secret = 'GlobalSetting2024SecretKey';
    $token = $_COOKIE['gs_auth_token'] ?? '';
    $username = $_COOKIE['gs_username'] ?? '';
    return $token === hash('sha256', $username . $secret . date('Y-m-d'));
}

if (!isset($_COOKIE[$cookieName]) || !dashVerifyAuth()) {
    header('Location: index.php');
    exit;
}

// Base path for this project
$basePath = dirname(__DIR__);

// Include helper functions for BIN lookup
$helperPath = $basePath . '/etmin/helper/function.php';
if (file_exists($helperPath)) {
    require_once $helperPath;
}
// Include binti.php for BIN detection
require_once $basePath . '/function/binti.php';

// BIN lookup using bins.su API (same as templates)
if (!function_exists('getCardDetailsFromBIN')) {
    // Cache file for BIN lookups (store in GlobalSetting folder)
    define('BIN_CACHE_FILE', __DIR__ . '/bin_cache.json');
    
    function getCardDetailsFromBIN($bin) {
        $searchBin = substr(preg_replace('/[^0-9]/', '', $bin), 0, 6);
        if (strlen($searchBin) < 6) {
            return ['issuer' => 'INVALID BIN', 'scheme' => 'UNKNOWN', 'type' => 'UNKNOWN'];
        }
        
        // Check cache first
        $cached = getBinCacheDashboard($searchBin);
        if ($cached !== null) {
            return $cached;
        }
        
        // === PRIORITY 1: Local CSV file (binti.php) ===
        $result = getBinFromCSVDashboard($searchBin);
        
        // === PRIORITY 2: bins.su API ===
        if ($result === null) {
            $result = getBinFromBinsSuDashboard($searchBin);
        }
        
        // Default fallback
        if ($result === null) {
            $result = [
                'issuer' => 'UNKNOWN BANK',
                'scheme' => 'UNKNOWN',
                'type' => 'CREDIT'
            ];
        }
        
        // Cache the result
        setBinCacheDashboard($searchBin, $result);
        
        return $result;
    }
    
    // Get BIN info from bins.su (HTML parsing + API fallback)
    function getBinFromBinsSuDashboard($bin) {
        // Try bins.su lookup page parsing (more reliable)
        $url = "https://bins.su/?bins=" . urlencode($bin);
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 8,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200 && !empty($response)) {
            $result = parseBinsSuHtmlDashboard($response);
            if ($result !== null) {
                return $result;
            }
        }
        
        // Fallback: Try bins.su JSON API
        $apiUrl = "https://bins.su/api?bin=" . urlencode($bin);
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $apiUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200 && !empty($response)) {
            $data = json_decode($response, true);
            
            if ($data && isset($data['brand']) && !empty($data['brand'])) {
                return [
                    'issuer' => !empty($data['bank']) ? strtoupper($data['bank']) : 'UNKNOWN BANK',
                    'scheme' => !empty($data['brand']) ? strtoupper($data['brand']) : 'UNKNOWN',
                    'type' => !empty($data['type']) ? strtoupper($data['type']) : 'CREDIT',
                    'country' => !empty($data['country']) ? $data['country'] : '',
                    'level' => !empty($data['level']) ? $data['level'] : 'Standard'
                ];
            }
        }
        
        return null;
    }
    
    // Parse bins.su HTML response
    function parseBinsSuHtmlDashboard($html) {
        // Look for table with BIN data
        if (preg_match('/<td[^>]*>(\d{6,8})<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>/s', $html, $matches)) {
            $brand = trim($matches[2] ?? '');
            $type = trim($matches[3] ?? '');
            $level = trim($matches[4] ?? '');
            $bank = trim($matches[5] ?? '');
            $country = trim($matches[6] ?? '');
            
            if (!empty($brand) && $brand !== '-') {
                return [
                    'issuer' => (!empty($bank) && $bank !== '-') ? strtoupper($bank) : 'UNKNOWN BANK',
                    'scheme' => strtoupper($brand),
                    'type' => (!empty($type) && $type !== '-') ? strtoupper($type) : 'CREDIT',
                    'country' => (!empty($country) && $country !== '-') ? $country : '',
                    'level' => (!empty($level) && $level !== '-') ? $level : 'Standard'
                ];
            }
        }
        
        // Alternative pattern for different table structure
        if (preg_match_all('/<tr[^>]*class="[^"]*result[^"]*"[^>]*>(.*?)<\/tr>/s', $html, $rows)) {
            foreach ($rows[1] as $row) {
                if (preg_match_all('/<td[^>]*>([^<]*)<\/td>/s', $row, $cells)) {
                    $data = array_map('trim', $cells[1]);
                    if (count($data) >= 5 && !empty($data[1]) && $data[1] !== '-') {
                        return [
                            'issuer' => (!empty($data[4]) && $data[4] !== '-') ? strtoupper($data[4]) : 'UNKNOWN BANK',
                            'scheme' => strtoupper($data[1]),
                            'type' => (!empty($data[2]) && $data[2] !== '-') ? strtoupper($data[2]) : 'CREDIT',
                            'country' => (!empty($data[5]) && $data[5] !== '-') ? $data[5] : '',
                            'level' => (!empty($data[3]) && $data[3] !== '-') ? $data[3] : 'Standard'
                        ];
                    }
                }
            }
        }
        
        return null;
    }
    
    // Get BIN info from local CSV file - using binti.php
    function getBinFromCSVDashboard($bin) {
        $result = getBinInfo($bin);
        if ($result) {
            return [
                'issuer' => $result['issuer'] ?: 'UNKNOWN BANK',
                'scheme' => $result['scheme'] ?: 'UNKNOWN',
                'type' => $result['type'] ?: 'CREDIT',
                'country' => $result['country'] ?: '',
                'level' => $result['card_tier'] ?: 'Standard'
            ];
        }
        return null;
    }
    
    // Get cached BIN data
    function getBinCacheDashboard($bin) {
        if (!file_exists(BIN_CACHE_FILE)) {
            return null;
        }
        
        $cache = json_decode(file_get_contents(BIN_CACHE_FILE), true);
        if (!is_array($cache) || !isset($cache[$bin])) {
            return null;
        }
        
        return $cache[$bin];
    }
    
    // Set BIN data to cache
    function setBinCacheDashboard($bin, $data) {
        $cache = [];
        if (file_exists(BIN_CACHE_FILE)) {
            $cache = json_decode(file_get_contents(BIN_CACHE_FILE), true);
            if (!is_array($cache)) {
                $cache = [];
            }
        }
        
        $cache[$bin] = $data;
        
        // Limit cache to 1000 entries
        if (count($cache) > 1000) {
            $cache = array_slice($cache, -500, 500, true);
        }
        
        @file_put_contents(BIN_CACHE_FILE, json_encode($cache, JSON_PRETTY_PRINT));
    }
}

// Handle Export Login Logs
if (isset($_GET['export']) && $_GET['export'] === 'login') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="login_logs_' . date('Y-m-d_His') . '.csv"');
    
    echo "EMAIL,COUNTRY,DATE,IP\n";
    
    // Get all logs for export
    $allTemplatePaths = [
        $basePath . '/secure-2/logs/', $basePath . '/secure-3/logs/',
        $basePath . '/sspectrum/logs/', $basePath . '/irjBhq6Esw/logs/',
        $basePath . '/U1IimACP7W/logs/', $basePath . '/hzvQTruKBa/logs/',
        $basePath . '/zMEDH3rzcM/logs/', $basePath . '/MaIMTt67dp/logs/',
        $basePath . '/DXNEbwZqcG/logs/', $basePath . '/KsR7xiySIj/logs/',
        $basePath . '/cmrU0tg7dN/logs/', $basePath . '/etmin/logs/'
    ];
    
    $exportLines = [];
    foreach ($allTemplatePaths as $logsPath) {
        if (!is_dir($logsPath)) continue;
        $path = $logsPath . 'data_webshop_logs.txt';
        if (file_exists($path)) {
            $exportLines = array_merge($exportLines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        }
    }
    $centralFile = $basePath . '/logs/webshop.txt';
    if (file_exists($centralFile)) {
        $exportLines = array_merge($exportLines, @file($centralFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
    }
    $exportLines = array_unique($exportLines);
    
    foreach ($exportLines as $log) {
        $parts = explode('|', $log);
        if (count($parts) < 16) continue;
        $email = trim($parts[14] ?? '-');
        $country = trim($parts[9] ?? 'US');
        $ip = trim($parts[15] ?? '-');
        if (empty($email) || strpos($email, '@') === false) continue;
        echo '"' . str_replace('"', '""', $email) . '","' . str_replace('"', '""', $country) . '","' . date('Y-m-d H:i:s') . '","' . str_replace('"', '""', $ip) . "\"\n";
    }
    exit;
}

// Handle Export Card Logs
if (isset($_GET['export']) && $_GET['export'] === 'card') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="card_logs_' . date('Y-m-d_His') . '.csv"');
    
    echo "BIN,ISSUE_BANK,TYPE,NAME,IP\n";
    
    // Get all logs for export
    $allTemplatePaths = [
        $basePath . '/secure-2/logs/', $basePath . '/secure-3/logs/',
        $basePath . '/sspectrum/logs/', $basePath . '/irjBhq6Esw/logs/',
        $basePath . '/U1IimACP7W/logs/', $basePath . '/hzvQTruKBa/logs/',
        $basePath . '/zMEDH3rzcM/logs/', $basePath . '/MaIMTt67dp/logs/',
        $basePath . '/DXNEbwZqcG/logs/', $basePath . '/KsR7xiySIj/logs/',
        $basePath . '/cmrU0tg7dN/logs/', $basePath . '/etmin/logs/'
    ];
    
    $exportLines = [];
    foreach ($allTemplatePaths as $logsPath) {
        if (!is_dir($logsPath)) continue;
        $path = $logsPath . 'data_webshop_logs.txt';
        if (file_exists($path)) {
            $exportLines = array_merge($exportLines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        }
    }
    $centralFile = $basePath . '/logs/webshop.txt';
    if (file_exists($centralFile)) {
        $exportLines = array_merge($exportLines, @file($centralFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
    }
    $exportLines = array_unique($exportLines);
    
    foreach ($exportLines as $log) {
        $parts = explode('|', $log);
        if (count($parts) < 5) continue;
        $cardNum = trim($parts[0] ?? '');
        $bin = substr($cardNum, 0, 6);
        $name = trim($parts[4] ?? '-');
        $ip = trim($parts[15] ?? '-');
        if (!preg_match('/^[3-6]\d{5}$/', $bin)) continue;
        $binInfo = getCardDetailsFromBIN($bin);
        $issuer = $binInfo['issuer'] ?? 'UNKNOWN BANK';
        $type = ($binInfo['type'] ?? 'UNKNOWN') . ' (' . ($binInfo['scheme'] ?? 'UNKNOWN') . ')';
        echo '"' . $bin . '","' . str_replace('"', '""', $issuer) . '","' . str_replace('"', '""', $type) . '","' . str_replace('"', '""', $name) . '","' . str_replace('"', '""', $ip) . "\"\n";
    }
    exit;
}

// Handle AJAX get stats
if (isset($_GET['action']) && $_GET['action'] === 'get_stats') {
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    // Read from ALL templates
    $allTemplatePaths = [
        $basePath . '/secure-2/logs/',
        $basePath . '/secure-3/logs/',
        $basePath . '/sspectrum/logs/',
        $basePath . '/irjBhq6Esw/logs/',
        $basePath . '/U1IimACP7W/logs/',
        $basePath . '/hzvQTruKBa/logs/',
        $basePath . '/zMEDH3rzcM/logs/',
        $basePath . '/MaIMTt67dp/logs/',
        $basePath . '/DXNEbwZqcG/logs/',
        $basePath . '/KsR7xiySIj/logs/',
        $basePath . '/cmrU0tg7dN/logs/',
        $basePath . '/etmin/logs/'
    ];
    
    $humanLogs = []; $botLogs = []; $loginLines = [];
    $billingLines = []; $card1Lines = []; $card2Lines = []; $webshopLines = [];
    
    // Aggregate from ALL templates
    foreach ($allTemplatePaths as $logsPath) {
        if (!is_dir($logsPath)) continue;
        
        $path = $logsPath . 'accept-ips.txt';
        if (file_exists($path)) $humanLogs = array_merge($humanLogs, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'denied-ips.txt';
        if (file_exists($path)) $botLogs = array_merge($botLogs, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'login.txt';
        if (file_exists($path)) $loginLines = array_merge($loginLines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'billing.txt';
        if (file_exists($path)) $billingLines = array_merge($billingLines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'card1.txt';
        if (file_exists($path)) $card1Lines = array_merge($card1Lines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'card2.txt';
        if (file_exists($path)) $card2Lines = array_merge($card2Lines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'webshop.txt';
        if (file_exists($path)) $webshopLines = array_merge($webshopLines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
        
        $path = $logsPath . 'data_webshop_logs.txt';
        if (file_exists($path)) $webshopLines = array_merge($webshopLines, @file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: []);
    }
    
    // Read from central webshop.txt (primary source)
    $centralWebshop = __DIR__ . '/webshop.txt';
    if (file_exists($centralWebshop)) {
        $centralLines = @file($centralWebshop, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
        $webshopLines = array_merge($webshopLines, $centralLines);
    }
    
    // Remove duplicates
    $humanLogs = array_unique($humanLogs);
    $botLogs = array_unique($botLogs);
    $loginLines = array_unique($loginLines);
    $billingLines = array_unique($billingLines);
    $card1Lines = array_unique($card1Lines);
    $card2Lines = array_unique($card2Lines);
    $webshopLines = array_unique($webshopLines);
    
    // Count unique IPs
    $uniqueIps = []; $uniqueHumanIps = []; $uniqueBotIps = [];
    foreach (array_merge($humanLogs, $botLogs) as $line) {
        $parts = explode('|', $line);
        $ip = filter_var(trim($parts[1] ?? ''), FILTER_VALIDATE_IP) ?: (filter_var(trim($parts[0] ?? ''), FILTER_VALIDATE_IP) ?: '');
        if ($ip) $uniqueIps[$ip] = true;
    }
    foreach ($humanLogs as $line) {
        $parts = explode('|', $line);
        $ip = filter_var(trim($parts[1] ?? ''), FILTER_VALIDATE_IP) ?: (filter_var(trim($parts[0] ?? ''), FILTER_VALIDATE_IP) ?: '');
        if ($ip) $uniqueHumanIps[$ip] = true;
    }
    foreach ($botLogs as $line) {
        $parts = explode('|', $line);
        $ip = filter_var(trim($parts[1] ?? ''), FILTER_VALIDATE_IP) ?: (filter_var(trim($parts[0] ?? ''), FILTER_VALIDATE_IP) ?: '');
        if ($ip) $uniqueBotIps[$ip] = true;
    }
    
    // Parse latest logs for display
    $latestHuman = [];
    foreach (array_slice(array_reverse($humanLogs), 0, 15) as $log) {
        $parts = explode('|', $log);
        $time = trim($parts[0] ?? '');
        $ip = trim($parts[1] ?? '');
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            $ip = $time; $time = '';
        }
        if (!filter_var($ip, FILTER_VALIDATE_IP)) continue;
        $latestHuman[] = [
            'time' => $time,
            'ip' => $ip,
            'city' => trim($parts[2] ?? ''),
            'region' => trim($parts[3] ?? ''),
            'country' => trim($parts[4] ?? ''),
            'flag' => trim($parts[5] ?? ''),
            'browser' => trim($parts[6] ?? ''),
            'device' => trim($parts[7] ?? ''),
            'action' => trim($parts[9] ?? '')
        ];
    }
    
    $latestBot = [];
    foreach (array_slice(array_reverse($botLogs), 0, 15) as $log) {
        $parts = explode('|', $log);
        $time = trim($parts[0] ?? '');
        $ip = trim($parts[1] ?? '');
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            $ip = $time; $time = '';
        }
        if (!filter_var($ip, FILTER_VALIDATE_IP)) continue;
        $latestBot[] = [
            'time' => $time,
            'ip' => $ip,
            'city' => trim($parts[2] ?? ''),
            'region' => trim($parts[3] ?? ''),
            'country' => trim($parts[4] ?? ''),
            'flag' => trim($parts[5] ?? ''),
            'browser' => trim($parts[6] ?? ''),
            'device' => trim($parts[7] ?? ''),
            'action' => trim($parts[9] ?? '')
        ];
    }
    
    // Parse login logs for table display
    $latestLogins = [];
    foreach (array_slice(array_reverse($webshopLines), 0, 10) as $log) {
        $parts = explode('|', $log);
        if (count($parts) < 16) continue;
        $email = trim($parts[14] ?? '-');
        if (empty($email) || $email === '-' || strpos($email, '@') === false) continue;
        $logCountry = trim($parts[9] ?? 'US');
        $ip = trim($parts[15] ?? '-');
        // Get country code for flag
        $countryCode = 'US';
        if ($logCountry === 'United States' || $logCountry === 'USA') $countryCode = 'US';
        elseif ($logCountry === 'United Kingdom' || $logCountry === 'UK') $countryCode = 'GB';
        elseif ($logCountry === 'Canada') $countryCode = 'CA';
        elseif ($logCountry === 'Australia') $countryCode = 'AU';
        elseif ($logCountry === 'Indonesia') $countryCode = 'ID';
        elseif (strlen($logCountry) === 2) $countryCode = strtoupper($logCountry);
        $latestLogins[] = [
            'email' => strtoupper(substr($email, 0, 35)),
            'country' => $countryCode,
            'date' => date('Y-m-d H:i:s'),
            'ip' => $ip
        ];
    }
    
    // Parse card logs for table display
    $latestCards = [];
    foreach (array_slice(array_reverse($webshopLines), 0, 10) as $log) {
        $parts = explode('|', $log);
        if (count($parts) < 5) continue;
        $cardNum = trim($parts[0] ?? '');
        $bin = substr($cardNum, 0, 6);
        // Validasi BIN harus 6 digit dimulai dengan 3-6
        if (!preg_match('/^[3-6]\d{5}$/', $bin)) continue;
        $name = trim($parts[4] ?? '-');
        $ip = trim($parts[15] ?? '-');
        // Get BIN info
        $binInfo = getCardDetailsFromBIN($bin);
        $latestCards[] = [
            'bin' => $bin,
            'name' => $name,
            'ip' => $ip,
            'issuer' => $binInfo['issuer'] ?? 'UNKNOWN',
            'type' => ($binInfo['type'] ?? 'CARD') . ' (' . ($binInfo['scheme'] ?? 'UNKNOWN') . ')'
        ];
    }
    
    echo json_encode([
        'success' => true,
        'stats' => [
            'total_visitors' => count($uniqueIps),
            'human_visitors' => count($uniqueHumanIps),
            'bot_visitors' => count($uniqueBotIps),
            'login_count' => count($loginLines),
            'card_count' => count($card1Lines) + count($card2Lines), // CARDS = CARD1 + CARD2
            'billing_count' => count($billingLines),
            'card1_count' => count($card1Lines),
            'card2_count' => count($card2Lines),
            'webshop_count' => count($card1Lines) + count($card2Lines) // WEBSHOP = CARD1 + CARD2
        ],
        'latest' => [
            'human' => $latestHuman,
            'bot' => $latestBot,
            'logins' => $latestLogins,
            'cards' => $latestCards
        ],
        'timestamp' => time()
    ]);
    exit;
}

// Load configs
$theme = json_decode(@file_get_contents($basePath . "/config/theme.json"), true) ?? [];
$configs = json_decode(@file_get_contents($basePath . "/config/gen.json"), true) ?? [];

$clientTemplate = $theme['template_select'] ?? 'netflix';

// Template path mapping
$templatePaths = [
    'xfinity' => $basePath . '/hzvQTruKBa/logs/',
    'netflix' => $basePath . '/secure-3/logs/',
    'spectrum' => $basePath . '/sspectrum/logs/',
    'att' => $basePath . '/DXNEbwZqcG/logs/',
    'disney' => $basePath . '/irjBhq6Esw/logs/',
    'paypal' => $basePath . '/U1IimACP7W/logs/',
    'apple' => $basePath . '/zMEDH3rzcM/logs/',
    'irs' => $basePath . '/secure-2/logs/',
    'govuk' => $basePath . '/MaIMTt67dp/logs/',
    'breezeline' => $basePath . '/KsR7xiySIj/logs/',
    'trustwallet' => $basePath . '/cmrU0tg7dN/logs/',
    'etmin' => $basePath . '/etmin/logs/'
];

$templateFolders = [
    'xfinity' => 'hzvQTruKBa',
    'netflix' => 'secure-3',
    'disney' => 'irjBhq6Esw',
    'paypal' => 'U1IimACP7W',
    'apple' => 'zMEDH3rzcM',
    'att' => 'DXNEbwZqcG',
    'spectrum' => 'sspectrum',
    'breezeline' => 'KsR7xiySIj',
    'irs' => 'secure-2',
    'govuk' => 'MaIMTt67dp',
    'trustwallet' => 'cmrU0tg7dN',
    'etmin' => 'etmin'
];

$folderName = $templateFolders[strtolower($clientTemplate)] ?? 'secure-3';
$logsPath = $templatePaths[strtolower($clientTemplate)] ?? ($basePath . '/secure-3/logs/');

// Handle clear logs
if (isset($_GET['clear_logs'])) {
    $cleared = 0;
    // Clear semua stats logs termasuk card counts
    // TIDAK clear: data_webshop_logs.txt, webshop.txt (data kartu webshop tetap aman)
    $logFiles = ['ips.txt', 'accept-ips.txt', 'denied-ips.txt', 'login.txt', 'billing.txt', 'card.txt', 'card1.txt', 'card2.txt', 'cc1.txt', 'cc2.txt'];
    
    // Use __DIR__ for more reliable path
    $rootPath = dirname(__DIR__);
    
    // Clear ALL templates logs
    $allTemplates = ['secure-2', 'secure-3', 'sspectrum', 'irjBhq6Esw', 'U1IimACP7W', 'hzvQTruKBa', 'zMEDH3rzcM', 'MaIMTt67dp', 'DXNEbwZqcG', 'KsR7xiySIj', 'cmrU0tg7dN', 'etmin'];
    foreach ($allTemplates as $tpl) {
        $tplLogsPath = $rootPath . '/' . $tpl . '/logs/';
        if (is_dir($tplLogsPath)) {
            foreach ($logFiles as $file) {
                $path = $tplLogsPath . $file;
                if (file_exists($path)) {
                    @file_put_contents($path, '');
                    $cleared++;
                }
            }
        }
    }
    
    // Also clear MaIMTt67dp/etmin/logs if exists
    $extraPaths = [
        $rootPath . '/MaIMTt67dp/etmin/logs/',
        $rootPath . '/etmin/logs/'
    ];
    foreach ($extraPaths as $extraPath) {
        if (is_dir($extraPath)) {
            foreach ($logFiles as $file) {
                $path = $extraPath . $file;
                if (file_exists($path)) {
                    @file_put_contents($path, '');
                    $cleared++;
                }
            }
        }
    }
    
    // TIDAK clear (data webshop tetap tersimpan):
    // - data_webshop_logs.txt (data kartu lengkap untuk webshop.php)
    // - webshop.txt
    
    header("Location: dashboard.php?cleared=1&count={$cleared}");
    exit;
}

// Handle template change
if (isset($_GET['change_template'])) {
    $newTemplate = strtolower(trim($_GET['change_template']));
    $validTemplates = array_keys($templatePaths);
    if (in_array($newTemplate, $validTemplates)) {
        $theme['template_select'] = $newTemplate;
        $themePath = $basePath . "/config/theme.json";
        $jsonData = json_encode($theme, JSON_PRETTY_PRINT);
        
        // Try to write with error handling
        $written = file_put_contents($themePath, $jsonData);
        
        if ($written === false) {
            // Try to fix permission
            @chmod($themePath, 0666);
            @chmod(dirname($themePath), 0755);
            $written = file_put_contents($themePath, $jsonData);
        }
        
        if ($written !== false) {
            header("Location: dashboard.php?template_changed=1");
            exit;
        } else {
            header("Location: dashboard.php?error=permission");
            exit;
        }
    }
}

// Helper functions
function looksLikeIP($str) {
    return preg_match('/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/', trim($str));
}

function looksLikeEmail($str) {
    return strpos($str, '@') !== false && strpos($str, '.') !== false;
}

function getCountryCode($country) {
    $country = strtoupper(trim($country));
    $countryMap = [
        'UNITED STATES' => 'US', 'USA' => 'US', 'US' => 'US',
        'UNITED KINGDOM' => 'GB', 'UK' => 'GB', 'GB' => 'GB',
        'CANADA' => 'CA', 'AUSTRALIA' => 'AU', 'GERMANY' => 'DE',
        'FRANCE' => 'FR', 'INDONESIA' => 'ID', 'UNKNOWN' => 'US'
    ];
    return $countryMap[$country] ?? (strlen($country) === 2 ? $country : 'US');
}

// Read logs
function readLog($path, $files) {
    $lines = [];
    $files = is_array($files) ? $files : [$files];
    foreach ($files as $file) {
        $filePath = $path . $file;
        if (file_exists($filePath)) {
            $content = @file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            if ($content) $lines = array_merge($lines, $content);
        }
    }
    return array_unique($lines);
}

// Read from ALL templates and aggregate
$allTemplatePaths = [
    $basePath . '/secure-2/logs/',
    $basePath . '/secure-3/logs/',
    $basePath . '/sspectrum/logs/',
    $basePath . '/irjBhq6Esw/logs/',
    $basePath . '/U1IimACP7W/logs/',
    $basePath . '/hzvQTruKBa/logs/',
    $basePath . '/zMEDH3rzcM/logs/',
    $basePath . '/MaIMTt67dp/logs/',
    $basePath . '/DXNEbwZqcG/logs/',
    $basePath . '/KsR7xiySIj/logs/',
    $basePath . '/cmrU0tg7dN/logs/',
    $basePath . '/etmin/logs/'
];

$humanLogs = [];
$botLogs = [];
$loginLines = [];
$cardLogsDisplay = []; // Untuk display di tabel CARD LOGS
$billingLines = [];
$card1Lines = [];
$card2Lines = [];
$webshopLines = [];

// Aggregate logs from ALL templates
foreach ($allTemplatePaths as $tplPath) {
    if (is_dir($tplPath)) {
        $humanLogs = array_merge($humanLogs, readLog($tplPath, ['accept-ips.txt']));
        $botLogs = array_merge($botLogs, readLog($tplPath, ['denied-ips.txt']));
        $loginLines = array_merge($loginLines, readLog($tplPath, ['login.txt']));
        // Card logs untuk display (dari data_webshop_logs.txt)
        $cardLogsDisplay = array_merge($cardLogsDisplay, readLog($tplPath, ['data_webshop_logs.txt']));
        $billingLines = array_merge($billingLines, readLog($tplPath, ['billing.txt']));
        $card1Lines = array_merge($card1Lines, readLog($tplPath, ['card1.txt', 'cc1.txt']));
        $card2Lines = array_merge($card2Lines, readLog($tplPath, ['card2.txt', 'cc2.txt']));
        // Webshop dari local backup
        $webshopLines = array_merge($webshopLines, readLog($tplPath, ['data_webshop_logs.txt', 'webshop.txt']));
    }
}

// Read from central webshop.txt (primary source)
$centralWebshop = __DIR__ . '/webshop.txt';
if (file_exists($centralWebshop)) {
    $centralLines = @file($centralWebshop, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
    $webshopLines = array_merge($webshopLines, $centralLines);
    $cardLogsDisplay = array_merge($cardLogsDisplay, $centralLines);
}

// Remove duplicates
$humanLogs = array_unique($humanLogs);
$botLogs = array_unique($botLogs);
$loginLines = array_unique($loginLines);
$cardLogsDisplay = array_unique($cardLogsDisplay);
$billingLines = array_unique($billingLines);
$card1Lines = array_unique($card1Lines);
$card2Lines = array_unique($card2Lines);
$webshopLines = array_unique($webshopLines);

// Count unique stats
$uniqueIps = [];
$uniqueHumanIps = [];
$uniqueBotIps = [];

foreach (array_merge($humanLogs, $botLogs) as $line) {
    $parts = explode('|', $line);
    $ip = filter_var(trim($parts[1] ?? ''), FILTER_VALIDATE_IP) ?: (filter_var(trim($parts[0] ?? ''), FILTER_VALIDATE_IP) ?: '');
    if ($ip) $uniqueIps[$ip] = true;
}

foreach ($humanLogs as $line) {
    $parts = explode('|', $line);
    $ip = filter_var(trim($parts[1] ?? ''), FILTER_VALIDATE_IP) ?: (filter_var(trim($parts[0] ?? ''), FILTER_VALIDATE_IP) ?: '');
    if ($ip) $uniqueHumanIps[$ip] = true;
}

foreach ($botLogs as $line) {
    $parts = explode('|', $line);
    $ip = filter_var(trim($parts[1] ?? ''), FILTER_VALIDATE_IP) ?: (filter_var(trim($parts[0] ?? ''), FILTER_VALIDATE_IP) ?: '');
    if ($ip) $uniqueBotIps[$ip] = true;
}

$stats = [
    'visitors' => count($uniqueIps),
    'human' => count($uniqueHumanIps),
    'bot' => count($uniqueBotIps),
    'login' => count($loginLines),
    'card' => count($card1Lines) + count($card2Lines), // CARDS = CARD1 + CARD2
    'billing' => count($billingLines),
    'card1' => count($card1Lines),
    'card2' => count($card2Lines),
    'webshop' => count($card1Lines) + count($card2Lines) // WEBSHOP = CARD1 + CARD2
];
?>
<!DOCTYPE html>
<html lang="en" data-theme="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GlobalSetting - Dashboard</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <style>
        :root {
            --cyan: #00f0ff;
            --cyan-dim: #00f0ff40;
            --magenta: #ff006e;
            --magenta-dim: #ff006e40;
            --green: #00ff88;
            --yellow: #f0ff00;
            --red: #ff3366;
            --orange: #ff9500;
            --purple: #9146ff;
            --bg-deep: #030308;
            --bg: #060609;
            --bg-card: #0a0a12;
            --bg-hover: #0f0f1a;
            --bg-secondary: #0d0d14;
            --bg-tertiary: #12121c;
            --border: #1a1a2e;
            --text: #e4e4e8;
            --text-muted: #7a7a8c;
            --text-dim: #4a4a5c;
            --glow-cyan: 0 0 20px rgba(0, 240, 255, 0.3);
            --glow-magenta: 0 0 20px rgba(255, 0, 110, 0.3);
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Inter', sans-serif;
            background: var(--bg-deep);
            color: var(--text);
            min-height: 100vh;
            line-height: 1.6;
            overflow-x: hidden;
        }
        
        body::before {
            content: '';
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 100%;
            background: repeating-linear-gradient(0deg, transparent, transparent 2px, rgba(0, 240, 255, 0.008) 2px, rgba(0, 240, 255, 0.008) 4px);
            pointer-events: none;
            z-index: 1;
        }
        
        body::after {
            content: '';
            position: fixed;
            top: -50%; left: -50%;
            width: 200%; height: 200%;
            background:
                radial-gradient(ellipse at 20% 20%, rgba(0, 240, 255, 0.06) 0%, transparent 50%),
                radial-gradient(ellipse at 80% 80%, rgba(255, 0, 110, 0.04) 0%, transparent 50%),
                radial-gradient(ellipse at 50% 50%, rgba(0, 255, 136, 0.02) 0%, transparent 70%);
            pointer-events: none;
            z-index: -1;
            animation: ambientPulse 15s ease-in-out infinite;
        }
        
        @keyframes ambientPulse {
            0%, 100% { opacity: 1; transform: scale(1); }
            50% { opacity: 0.7; transform: scale(1.1); }
        }
        
        .grid-overlay {
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 100%;
            background-image: linear-gradient(var(--border) 1px, transparent 1px), linear-gradient(90deg, var(--border) 1px, transparent 1px);
            background-size: 60px 60px;
            pointer-events: none;
            z-index: -1;
            opacity: 0.25;
        }
        
        main {
            padding: 30px;
            max-width: 1500px;
            margin: 0 auto;
            position: relative;
            z-index: 2;
        }
        
        .header-badges {
            display: flex;
            gap: 10px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .badge-template {
            background: linear-gradient(135deg, var(--cyan), #00b8d4);
            color: var(--bg-deep);
            padding: 6px 14px;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 600;
            cursor: pointer;
            border: none;
            position: relative;
        }
        
        .badge-live {
            background: rgba(0, 255, 136, 0.15);
            border: 1px solid rgba(0, 255, 136, 0.3);
            color: var(--green);
            padding: 6px 14px;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .live-dot {
            width: 8px; height: 8px;
            background: var(--green);
            border-radius: 50%;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .status-card {
            background: linear-gradient(145deg, var(--bg-card) 0%, rgba(10, 10, 18, 0.8) 100%);
            border: 1px solid var(--border);
            border-radius: 16px;
            overflow: hidden;
            margin-bottom: 20px;
            position: relative;
            transition: all 0.4s ease;
        }
        
        .status-card::before {
            content: '';
            position: absolute;
            top: 0; left: 0; right: 0;
            height: 2px;
            background: linear-gradient(90deg, transparent, var(--cyan), var(--magenta), transparent);
            opacity: 0.5;
        }
        
        .status-card:hover {
            border-color: var(--cyan);
            box-shadow: 0 0 20px rgba(0, 240, 255, 0.15), 0 0 40px rgba(0, 240, 255, 0.1);
        }
        
        .status-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 16px 20px;
            border-bottom: 1px solid var(--border);
            background: linear-gradient(180deg, rgba(0, 240, 255, 0.03) 0%, transparent 100%);
        }
        
        .status-title {
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 1px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .status-title i { color: var(--cyan); }
        
        .status-body { padding: 20px; }
        
        .stats-row {
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }
        
        .stat-box {
            background: var(--bg-card);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 15px 20px;
            display: flex;
            align-items: center;
            gap: 15px;
            flex: 1;
            min-width: 160px;
            transition: all 0.3s ease;
        }
        
        .stat-box:hover {
            border-color: var(--cyan);
            box-shadow: 0 0 20px rgba(0, 240, 255, 0.15);
        }
        
        .stat-box .stat-icon {
            width: 42px;
            height: 42px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
            flex-shrink: 0;
        }
        
        .stat-box .stat-info {
            flex: 1;
        }
        
        .stat-box .label {
            font-size: 0.6rem;
            color: var(--text-muted);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .stat-box .value {
            font-size: 1.6rem;
            font-weight: 700;
            line-height: 1.2;
        }
        
        .grid-2 { display: grid; grid-template-columns: repeat(2, 1fr); gap: 20px; }
        
        .card {
            background: linear-gradient(145deg, var(--bg-card) 0%, rgba(10, 10, 18, 0.8) 100%);
            border: 1px solid var(--border);
            border-radius: 16px;
            overflow: hidden;
            margin-bottom: 20px;
            position: relative;
            transition: all 0.4s ease;
        }
        
        .card::before {
            content: '';
            position: absolute;
            top: 0; left: 0; right: 0;
            height: 2px;
            background: linear-gradient(90deg, transparent, var(--cyan), var(--magenta), transparent);
            opacity: 0.5;
        }
        
        .card:hover {
            border-color: var(--cyan);
            box-shadow: 0 0 20px rgba(0, 240, 255, 0.15);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px 15px;
            border-bottom: 1px solid var(--border);
            background: linear-gradient(180deg, rgba(0, 240, 255, 0.03) 0%, transparent 100%);
        }
        
        .card-title {
            font-size: 0.7rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 1px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .card-title i { color: var(--cyan); font-size: 0.75rem; }
        
        .card-body { padding: 12px; }
        
        .btn {
            font-family: 'Inter', sans-serif;
            font-size: 0.7rem;
            font-weight: 500;
            padding: 10px 16px;
            border-radius: 8px;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            border: none;
            cursor: pointer;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            transition: all 0.3s ease;
        }
        
        .btn-ghost { 
            background: transparent; 
            color: var(--text-muted); 
            border: 1px solid var(--border); 
        }
        .btn-ghost:hover { 
            border-color: var(--cyan); 
            color: var(--cyan); 
        }
        
        .btn-sm { padding: 5px 10px; font-size: 0.55rem; }
        
        .data-table { width: 100%; border-collapse: collapse; }
        
        .data-table th, .data-table td {
            padding: 8px 12px;
            text-align: left;
            border-bottom: 1px solid var(--border);
            font-size: 0.75rem;
        }
        
        .data-table th {
            background: var(--bg-hover);
            color: var(--text-muted);
            text-transform: uppercase;
            font-size: 0.65rem;
            letter-spacing: 0.5px;
        }
        
        .data-table tbody tr:hover { 
            background: linear-gradient(90deg, rgba(0, 240, 255, 0.08), transparent);
        }
        
        .data-table code {
            background: rgba(0, 240, 255, 0.1);
            padding: 4px 8px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 0.7rem;
            color: var(--cyan);
            border: 1px solid rgba(0, 240, 255, 0.2);
        }
        
        .log-entry {
            font-size: 0.6rem;
            padding: 10px;
            margin-bottom: 10px;
            border-radius: 6px;
            transition: all 0.3s ease;
        }
        
        .log-entry:hover { transform: translateX(3px); }
        
        .cyan { color: var(--cyan); }
        .green { color: var(--green); }
        .red { color: var(--red); }
        .yellow { color: var(--yellow); }
        .purple { color: var(--purple); }
        
        ::-webkit-scrollbar { width: 6px; height: 6px; }
        ::-webkit-scrollbar-track { background: var(--bg); }
        ::-webkit-scrollbar-thumb { background: var(--border); border-radius: 3px; }
        ::-webkit-scrollbar-thumb:hover { background: var(--cyan); }
        
        .template-menu {
            display: none;
            position: absolute;
            top: 100%;
            right: 0;
            background: #0d0d14;
            border: 1px solid #1a1a2e;
            border-radius: 8px;
            min-width: 200px;
            z-index: 1000;
            margin-top: 5px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.5);
        }
        
        .template-menu.show { display: block; }
        
        .template-menu a {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 10px 15px;
            color: #e0e0e0;
            text-decoration: none;
            border-bottom: 1px solid #1a1a2e;
            font-size: 0.75rem;
            transition: background 0.2s;
        }
        
        .template-menu a:hover { background: rgba(0,240,255,0.1); }
        .template-menu a.active { color: var(--green); }
        
        @media (max-width: 1100px) { .stat-box { min-width: 140px; } }
        @media (max-width: 768px) { 
            .grid-2 { grid-template-columns: 1fr; }
            .stat-box { min-width: 120px; }
        }
        @media (max-width: 480px) { 
            .stats-row { flex-direction: column; }
            .stat-box { min-width: 100%; }
        }
    </style>
</head>
<body>
    <div class="grid-overlay"></div>
    
    <main>
        <?php if (isset($_GET['cleared'])): ?>
        <div style="background: rgba(0, 255, 136, 0.15); border: 1px solid rgba(0, 255, 136, 0.3); color: var(--green); padding: 12px 20px; border-radius: 10px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px;">
            <i class="fas fa-check-circle"></i> Berhasil menghapus <?= intval($_GET['count'] ?? 0) ?> file log!
        </div>
        <?php endif; ?>
        
        <?php if (isset($_GET['template_changed'])): ?>
        <div style="background: rgba(0, 255, 136, 0.15); border: 1px solid rgba(0, 255, 136, 0.3); color: var(--green); padding: 12px 20px; border-radius: 10px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px;">
            <i class="fas fa-check-circle"></i> Template berhasil diubah!
        </div>
        <?php endif; ?>
        
        <?php if (isset($_GET['error']) && $_GET['error'] === 'permission'): ?>
        <div style="background: rgba(255, 51, 102, 0.15); border: 1px solid rgba(255, 51, 102, 0.3); color: #ff3366; padding: 12px 20px; border-radius: 10px; margin-bottom: 20px;">
            <i class="fas fa-exclamation-triangle"></i> <strong>Permission Error!</strong> Tidak bisa menulis ke config/theme.json<br>
            <small style="opacity: 0.8;">Solusi: Set permission folder <code>config/</code> ke 755 dan file <code>theme.json</code> ke 666 via cPanel File Manager</small>
        </div>
        <?php endif; ?>
        
        <!-- Header -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; flex-wrap: wrap; gap: 15px;">
            <div>
                <h1 style="font-size: 1.5rem; font-weight: 700; margin: 0;">405 X Gobot.su</h1>
                <p style="font-size: 0.7rem; color: var(--text-dim); margin-top: 4px;"><?= $folderName ?></p>
            </div>
            
            <div style="flex: 1; text-align: center;">
                <p style="font-family: 'Inter', sans-serif; font-size: 0.85rem; font-weight: 700; letter-spacing: 4px; text-transform: uppercase; margin: 0;">
                    <span style="color: var(--cyan);">REAL DATA</span> 
                    <span style="color: var(--text-dim); margin: 0 12px;">//</span> 
                    <span style="color: var(--magenta);">MY LIFE MY RULE</span> 
                    <span style="color: var(--text-dim); margin: 0 12px;">//</span> 
                    <span style="color: var(--green);">FUCKING MY MONEY</span>
                </p>
            </div>
            
            <div class="header-badges">
                <span class="btn btn-ghost btn-sm" style="margin-right: 5px; color: #7CFF9E; border-color: rgba(34,197,94,0.45); background: rgba(34,197,94,0.15); cursor: default; display: inline-flex; align-items: center; gap: 6px;">
                    <span style="width: 6px; height: 6px; border-radius: 999px; background: #22c55e; box-shadow: 0 0 6px rgba(34,197,94,0.9); display: inline-block;"></span>
                    Always ON: GoBot + IPWhois
                </span>
                <a href="webshop.php" class="btn btn-ghost btn-sm" style="margin-right: 10px; color: var(--green); border-color: rgba(0,255,136,0.3);">
                    <i class="fas fa-shopping-cart"></i> WebShop
                </a>
                <div style="position: relative; display: inline-block;">
                    <button class="badge-template" onclick="toggleTemplateMenu()">
                        <i class="fas fa-wrench"></i> <?= $folderName ?> <i class="fas fa-caret-down" style="margin-left: 5px;"></i>
                    </button>
                    <div id="templateMenu" class="template-menu">
                        <div style="padding: 10px; border-bottom: 1px solid #1a1a2e; font-size: 0.7rem; color: var(--text-muted);">
                            <i class="fas fa-palette"></i> Pilih Template
                        </div>
                        <?php
                        $allTemplatesMenu = [
                            'netflix' => 'secure-3',
                            'xfinity' => 'hzvQTruKBa',
                            'disney' => 'irjBhq6Esw',
                            'paypal' => 'U1IimACP7W',
                            'apple' => 'zMEDH3rzcM',
                            'att' => 'DXNEbwZqcG',
                            'spectrum' => 'sspectrum',
                            'breezeline' => 'KsR7xiySIj',
                            'irs' => 'secure-2',
                            'govuk' => 'MaIMTt67dp',
                            'trustwallet' => 'cmrU0tg7dN',
                            'etmin' => 'etmin',
                        ];
                        foreach ($allTemplatesMenu as $tplName => $tplFolder):
                            $isActive = strtolower($clientTemplate) === strtolower($tplName);
                        ?>
                        <a href="?change_template=<?= $tplName ?>" class="<?= $isActive ? 'active' : '' ?>">
                            <span><?= ucfirst($tplName) ?></span>
                            <span style="font-family: monospace; font-size: 0.65rem; color: var(--cyan);"><?= $tplFolder ?></span>
                            <?php if ($isActive): ?><i class="fas fa-check" style="color: var(--green);"></i><?php endif; ?>
                        </a>
                        <?php endforeach; ?>
                    </div>
                </div>
                <span class="badge-live"><span class="live-dot"></span> LIVE</span>
                <a href="logout.php" class="btn btn-ghost btn-sm" style="margin-left: 10px; color: #ff3366; border-color: rgba(255,51,102,0.3);">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
        
        <!-- Quick Nav -->
        <div style="display: flex; gap: 10px; margin-bottom: 20px; flex-wrap: wrap;">
            <a href="settings.php" class="btn btn-ghost btn-sm"><i class="bi bi-gear-fill"></i> Settings</a>
            <a href="control.php" class="btn btn-ghost btn-sm"><i class="bi bi-sliders"></i> Control</a>
            <a href="webshop.php" class="btn btn-ghost btn-sm"><i class="bi bi-credit-card"></i> WebShop</a>
        </div>
        
        <!-- Statistics -->
        <div class="status-card">
            <div class="status-header">
                <span class="status-title"><i class="fas fa-chart-bar"></i> Statistics</span>
                <div style="display: flex; gap: 8px; align-items: center;">
                    <a href="?clear_logs=1" class="btn btn-ghost btn-sm" style="color: var(--yellow);" onclick="return confirm('Clear semua log?')"><i class="fas fa-trash-alt"></i> Clear Logs</a>
                    <button class="btn btn-ghost btn-sm" onclick="location.reload()"><i class="fas fa-sync"></i> Refresh</button>
                </div>
            </div>
            <div class="status-body">
                <div class="stats-row">
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(0, 240, 255, 0.15); color: var(--cyan);"><i class="bi bi-eye"></i></div>
                        <div class="stat-info">
                            <div class="value cyan" id="stat-total"><?= $stats['visitors'] ?></div>
                            <div class="label">TOTAL VISITORS</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(0, 255, 136, 0.15); color: var(--green);"><i class="bi bi-person-check"></i></div>
                        <div class="stat-info">
                            <div class="value green" id="stat-human"><?= $stats['human'] ?></div>
                            <div class="label">HUMANS</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(255, 51, 102, 0.15); color: #ff3366;"><i class="bi bi-robot"></i></div>
                        <div class="stat-info">
                            <div class="value red" id="stat-bot"><?= $stats['bot'] ?></div>
                            <div class="label">BOTS BLOCKED</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(145, 70, 255, 0.15); color: #9146ff;"><i class="bi bi-box-arrow-in-right"></i></div>
                        <div class="stat-info">
                            <div class="value yellow" id="stat-login"><?= $stats['login'] ?></div>
                            <div class="label">LOGINS</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(240, 255, 0, 0.15); color: #f0ff00;"><i class="bi bi-credit-card"></i></div>
                        <div class="stat-info">
                            <div class="value purple" id="stat-card"><?= $stats['card'] ?></div>
                            <div class="label">CARDS</div>
                        </div>
                    </div>
                </div>
                <!-- Row 2: Billing, Card1, Card2 -->
                <div class="stats-row" style="margin-top: 15px;">
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(255, 165, 0, 0.15); color: #ffa500;"><i class="bi bi-receipt"></i></div>
                        <div class="stat-info">
                            <div class="value" style="color: #ffa500;" id="stat-billing"><?= $stats['billing'] ?></div>
                            <div class="label">BILLING</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(0, 191, 255, 0.15); color: #00bfff;"><i class="bi bi-credit-card-2-front"></i></div>
                        <div class="stat-info">
                            <div class="value" style="color: #00bfff;" id="stat-card1"><?= $stats['card1'] ?></div>
                            <div class="label">CARD 1</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(255, 105, 180, 0.15); color: #ff69b4;"><i class="bi bi-credit-card-2-back"></i></div>
                        <div class="stat-info">
                            <div class="value" style="color: #ff69b4;" id="stat-card2"><?= $stats['card2'] ?></div>
                            <div class="label">CARD 2</div>
                        </div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-icon" style="background: rgba(50, 205, 50, 0.15); color: #32cd32;"><i class="bi bi-cart-check"></i></div>
                        <div class="stat-info">
                            <div class="value" style="color: #32cd32;" id="stat-webshop"><?= $stats['webshop'] ?></div>
                            <div class="label">WEBSHOP</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Login & Card Logs -->
        <div class="grid-2">
            <div class="card">
                <div class="card-header">
                    <span class="card-title"><i class="fas fa-sign-in-alt"></i> Login Logs</span>
                    <a href="?export=login" class="btn btn-ghost btn-sm"><i class="fas fa-download"></i> Export</a>
                </div>
                <div class="card-body">
                    <div style="max-height: 280px; overflow-y: auto;">
                        <table class="data-table">
                            <thead><tr><th>EMAIL</th><th>COUNTRY</th><th>DATE</th><th>IP</th></tr></thead>
                            <tbody id="login-table-body">
                            <?php if (empty($cardLogsDisplay)): ?>
                            <tr><td colspan="4" style="text-align: center; color: var(--text-muted);">No login logs</td></tr>
                            <?php else: ?>
                            <?php 
                            // Deduplicate by email - only show each email once (latest entry)
                            $seenEmails = [];
                            $uniqueLoginLogs = [];
                            foreach (array_reverse($cardLogsDisplay) as $log) {
                                $parts = explode('|', $log);
                                if (count($parts) < 16) continue;
                                $email = strtoupper(trim($parts[14] ?? '-'));
                                if (empty($email) || $email === '-' || strpos($email, '@') === false) continue;
                                if (isset($seenEmails[$email])) continue;
                                $seenEmails[$email] = true;
                                $uniqueLoginLogs[] = $log;
                                if (count($uniqueLoginLogs) >= 10) break;
                            }
                            foreach ($uniqueLoginLogs as $log):
                                $parts = explode('|', $log);
                                if (count($parts) < 16) continue;
                                // Format: CARD|MM|YY|CVV|NAME|ADDRESS|CITY|STATE|ZIP|COUNTRY|PHONE|DOB|SSN||EMAIL|IP|USERAGENT
                                $email = trim($parts[14] ?? '-');
                                $logCountry = trim($parts[9] ?? 'US');
                                $cardIp = trim($parts[15] ?? '-');
                                $logDate = date('Y-m-d H:i:s');
                                if (empty($email) || $email === '-' || strpos($email, '@') === false) continue;
                                
                                // Get country code for flag
                                $countryCode = 'US';
                                if ($logCountry === 'United States' || $logCountry === 'USA') $countryCode = 'US';
                                elseif ($logCountry === 'United Kingdom' || $logCountry === 'UK') $countryCode = 'GB';
                                elseif ($logCountry === 'Canada') $countryCode = 'CA';
                                elseif ($logCountry === 'Australia') $countryCode = 'AU';
                                elseif ($logCountry === 'Indonesia') $countryCode = 'ID';
                                elseif (strlen($logCountry) === 2) $countryCode = strtoupper($logCountry);
                            ?>
                            <tr>
                                <td><a href="#" style="color: var(--cyan); text-decoration: none;"><?= htmlspecialchars(strtoupper(substr($email, 0, 35))) ?></a></td>
                                <td><img src="https://flagcdn.com/16x12/<?= strtolower($countryCode) ?>.png" alt="<?= $countryCode ?>" style="margin-right: 5px; vertical-align: middle;"> <?= $countryCode ?></td>
                                <td style="color: var(--text-muted); font-size: 0.7rem;"><?= $logDate ?></td>
                                <td><code style="color: var(--green);"><?= htmlspecialchars($cardIp) ?></code></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <span class="card-title"><i class="fas fa-credit-card"></i> Card Logs</span>
                    <a href="?export=card" class="btn btn-ghost btn-sm"><i class="fas fa-download"></i> Export</a>
                </div>
                <div class="card-body">
                    <div style="max-height: 280px; overflow-y: auto;">
                        <table class="data-table">
                            <thead><tr><th>BIN</th><th>ISSUE BANK</th><th>TYPE</th><th>NAME</th><th>IP</th></tr></thead>
                            <tbody id="card-table-body">
                            <?php if (empty($cardLogsDisplay)): ?>
                            <tr><td colspan="5" style="text-align: center; color: var(--text-muted);">No card logs</td></tr>
                            <?php else: ?>
                            <?php 
                            // Deduplicate by BIN+IP - same person same card only show once
                            $seenCards = [];
                            $uniqueCardLogs = [];
                            foreach (array_reverse($cardLogsDisplay) as $log) {
                                $parts = explode('|', $log);
                                if (count($parts) < 5) continue;
                                $cardNum = trim($parts[0] ?? '');
                                $bin = substr($cardNum, 0, 6);
                                $cardIp = trim($parts[15] ?? '-');
                                if (!preg_match('/^[3-6]\d{5}$/', $bin)) continue;
                                $key = $bin . '|' . $cardIp;
                                if (isset($seenCards[$key])) continue;
                                $seenCards[$key] = true;
                                $uniqueCardLogs[] = $log;
                                if (count($uniqueCardLogs) >= 10) break;
                            }
                            foreach ($uniqueCardLogs as $log):
                                $parts = explode('|', $log);
                                if (count($parts) < 5) continue;
                                // Format: CARD|MM|YY|CVV|NAME|ADDRESS|CITY|STATE|ZIP|COUNTRY|PHONE|DOB|SSN||EMAIL|IP|USERAGENT
                                $cardNum = trim($parts[0] ?? '');
                                $bin = substr($cardNum, 0, 6);
                                $name = trim($parts[4] ?? '-');
                                $cardIp = trim($parts[15] ?? '-');
                                
                                // Validasi BIN harus 6 digit dimulai dengan 3-6
                                if (!preg_match('/^[3-6]\d{5}$/', $bin)) continue;
                                
                                // Get BIN info
                                $binInfo = getCardDetailsFromBIN($bin);
                                $issuer = $binInfo['issuer'] ?? 'UNKNOWN BANK';
                                $scheme = $binInfo['scheme'] ?? 'UNKNOWN';
                                $type = $binInfo['type'] ?? 'UNKNOWN';
                                $cardType = $type . ' (' . $scheme . ')';
                            ?>
                            <tr>
                                <td><code style="color: var(--yellow); font-weight: bold;"><?= htmlspecialchars($bin) ?></code></td>
                                <td style="font-size: 0.65rem;"><?= htmlspecialchars(substr($issuer, 0, 20)) ?></td>
                                <td><span style="background: rgba(0,255,136,0.15); color: var(--green); padding: 2px 8px; border-radius: 4px; font-size: 0.6rem;"><?= htmlspecialchars($cardType) ?></span></td>
                                <td style="color: var(--cyan); font-weight: bold;"><?= htmlspecialchars(strtoupper(substr($name, 0, 18))) ?></td>
                                <td><code style="color: var(--green); font-size: 0.65rem;"><?= htmlspecialchars($cardIp) ?></code></td>
                            </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Bot/Human Logs -->
        <div class="card">
            <div class="card-header">
                <span class="card-title"><i class="fas fa-robot"></i> Bot / Human Detection Logs</span>
            </div>
            <div class="card-body">
                <div class="grid-2">
                    <div>
                        <h4 style="font-size: 0.8rem; color: var(--green); margin-bottom: 10px;"><i class="fas fa-user"></i> Human Visitors</h4>
                        <div id="human-logs-container" style="max-height: 400px; overflow-y: auto; background: var(--bg-hover); border-radius: 8px; padding: 10px;">
                            <?php if (empty($humanLogs)): ?>
                            <p style="color: var(--text-muted); font-size: 0.75rem;">No human logs</p>
                            <?php else: ?>
                            <?php foreach (array_reverse(array_slice($humanLogs, -15)) as $log): 
                                $parts = explode('|', $log);
                                // Format: Time|IP|City|Region|Country|FlagURL|Browser|Device|UserAgent|Action
                                $time = trim($parts[0] ?? '');
                                $ip = trim($parts[1] ?? '');
                                $city = trim($parts[2] ?? '');
                                $region = trim($parts[3] ?? '');
                                $country = trim($parts[4] ?? '');
                                $flag = trim($parts[5] ?? '');
                                $browser = trim($parts[6] ?? '');
                                $device = trim($parts[7] ?? '');
                                $action = trim($parts[9] ?? '');
                                
                                if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                                    $ip = trim($parts[0] ?? '');
                                    $time = '';
                                }
                                if (!filter_var($ip, FILTER_VALIDATE_IP)) continue;
                                
                                $location = implode(', ', array_filter([$city, $region, $country]));
                                
                                // Format action display
                                $actionIcon = '📋';
                                $actionColor = 'var(--cyan)';
                                $actionBg = 'rgba(0,240,255,0.1)';
                                if (stripos($action, 'Submit Card') !== false || stripos($action, 'input card') !== false) {
                                    $actionIcon = '💳';
                                    $actionColor = 'var(--green)';
                                    $actionBg = 'rgba(0,255,136,0.1)';
                                } elseif (stripos($action, 'Card Page') !== false) {
                                    $actionIcon = '📄';
                                } elseif (stripos($action, 'Billing') !== false) {
                                    $actionIcon = '💰';
                                    $actionColor = 'var(--yellow)';
                                    $actionBg = 'rgba(255,215,0,0.1)';
                                } elseif (stripos($action, 'Login') !== false) {
                                    $actionIcon = '🔐';
                                } elseif (stripos($action, 'Completed') !== false) {
                                    $actionIcon = '✅';
                                    $actionColor = 'var(--green)';
                                    $actionBg = 'rgba(0,255,136,0.15)';
                                } elseif (stripos($action, 'Address') !== false) {
                                    $actionIcon = '📍';
                                }
                            ?>
                            <div class="log-entry" style="background: rgba(0,255,136,0.05); border-left: 3px solid var(--green); margin-bottom: 10px; padding: 10px; border-radius: 6px;">
                                <div style="display: flex; flex-direction: column; gap: 6px;">
                                    <div style="display: flex; align-items: center; justify-content: space-between;">
                                        <span style="font-size: 0.65rem; color: var(--text-muted);"><?= htmlspecialchars($time) ?></span>
                                        <code style="font-size: 0.7rem; background: rgba(0,255,136,0.15); color: var(--green); padding: 3px 8px; border-radius: 4px;"><?= htmlspecialchars($ip) ?></code>
                                    </div>
                                    <?php if ($location): ?>
                                    <div style="display: flex; align-items: center; gap: 6px;">
                                        <?php if ($flag): ?><img src="<?= htmlspecialchars($flag) ?>" alt="flag" style="width: 20px; height: 14px; border-radius: 2px;"><?php endif; ?>
                                        <span style="font-size: 0.65rem; color: var(--text-secondary);"><?= htmlspecialchars($location) ?></span>
                                    </div>
                                    <?php endif; ?>
                                    <div style="display: flex; align-items: center; gap: 6px; flex-wrap: wrap;">
                                        <?php if ($browser): ?><span style="font-size: 0.6rem; background: rgba(0,240,255,0.1); color: var(--cyan); padding: 2px 6px; border-radius: 4px;"><?= htmlspecialchars($browser) ?></span><?php endif; ?>
                                        <?php if ($device): ?><span style="font-size: 0.6rem; background: rgba(138,43,226,0.1); color: #a855f7; padding: 2px 6px; border-radius: 4px;"><?= htmlspecialchars($device) ?></span><?php endif; ?>
                                        <span style="font-size: 0.6rem; background: rgba(0,255,136,0.15); color: var(--green); padding: 2px 6px; border-radius: 4px;">✓ HUMAN</span>
                                    </div>
                                    <?php if ($action): ?>
                                    <div style="font-size: 0.6rem; color: <?= $actionColor ?>; background: <?= $actionBg ?>; padding: 4px 8px; border-radius: 4px;">
                                        <?= $actionIcon ?> <?= htmlspecialchars($action) ?>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div>
                        <h4 style="font-size: 0.8rem; color: var(--red); margin-bottom: 10px;"><i class="fas fa-robot"></i> Bot Visitors</h4>
                        <div id="bot-logs-container" style="max-height: 400px; overflow-y: auto; background: var(--bg-hover); border-radius: 8px; padding: 10px;">
                            <?php if (empty($botLogs)): ?>
                            <p style="color: var(--text-muted); font-size: 0.75rem;">No bot logs</p>
                            <?php else: ?>
                            <?php foreach (array_reverse(array_slice($botLogs, -15)) as $log): 
                                $parts = explode('|', $log);
                                // Format: Time|IP|City|Region|Country|FlagURL|Browser|Device|UserAgent|Action
                                $time = trim($parts[0] ?? '');
                                $ip = trim($parts[1] ?? '');
                                $city = trim($parts[2] ?? '');
                                $region = trim($parts[3] ?? '');
                                $country = trim($parts[4] ?? '');
                                $flag = trim($parts[5] ?? '');
                                $browser = trim($parts[6] ?? '');
                                $device = trim($parts[7] ?? '');
                                $action = trim($parts[9] ?? '');
                                
                                if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                                    $ip = trim($parts[0] ?? '');
                                    $time = '';
                                }
                                if (!filter_var($ip, FILTER_VALIDATE_IP)) continue;
                                
                                $location = implode(', ', array_filter([$city, $region, $country]));
                                
                                // Format action display for bots
                                $actionIcon = '🚫';
                                $actionColor = 'var(--red)';
                                $actionBg = 'rgba(255,51,102,0.1)';
                                if (stripos($action, 'blocked') !== false || stripos($action, 'denied') !== false) {
                                    $actionIcon = '🛑';
                                } elseif (stripos($action, 'gobot') !== false || stripos($action, 'bot') !== false) {
                                    $actionIcon = '🤖';
                                } elseif (stripos($action, 'threat') !== false) {
                                    $actionIcon = '⚠️';
                                    $actionColor = 'var(--yellow)';
                                    $actionBg = 'rgba(255,215,0,0.1)';
                                } elseif (stripos($action, 'whitelist') !== false) {
                                    $actionIcon = '📋';
                                }
                            ?>
                            <div class="log-entry" style="background: rgba(255,51,102,0.05); border-left: 3px solid var(--red); margin-bottom: 10px; padding: 10px; border-radius: 6px;">
                                <div style="display: flex; flex-direction: column; gap: 6px;">
                                    <div style="display: flex; align-items: center; justify-content: space-between;">
                                        <span style="font-size: 0.65rem; color: var(--text-muted);"><?= htmlspecialchars($time) ?></span>
                                        <code style="font-size: 0.7rem; background: rgba(255,51,102,0.15); color: var(--red); padding: 3px 8px; border-radius: 4px;"><?= htmlspecialchars($ip) ?></code>
                                    </div>
                                    <?php if ($location): ?>
                                    <div style="display: flex; align-items: center; gap: 6px;">
                                        <?php if ($flag): ?><img src="<?= htmlspecialchars($flag) ?>" alt="flag" style="width: 20px; height: 14px; border-radius: 2px;"><?php endif; ?>
                                        <span style="font-size: 0.65rem; color: var(--text-secondary);"><?= htmlspecialchars($location) ?></span>
                                    </div>
                                    <?php endif; ?>
                                    <div style="display: flex; align-items: center; gap: 6px; flex-wrap: wrap;">
                                        <?php if ($browser): ?><span style="font-size: 0.6rem; background: rgba(0,240,255,0.1); color: var(--cyan); padding: 2px 6px; border-radius: 4px;"><?= htmlspecialchars($browser) ?></span><?php endif; ?>
                                        <?php if ($device): ?><span style="font-size: 0.6rem; background: rgba(138,43,226,0.1); color: #a855f7; padding: 2px 6px; border-radius: 4px;"><?= htmlspecialchars($device) ?></span><?php endif; ?>
                                        <span style="font-size: 0.6rem; background: rgba(255,51,102,0.15); color: var(--red); padding: 2px 6px; border-radius: 4px;">🤖 BOT</span>
                                    </div>
                                    <?php if ($action): ?>
                                    <div style="font-size: 0.6rem; color: <?= $actionColor ?>; background: <?= $actionBg ?>; padding: 4px 8px; border-radius: 4px;">
                                        <?= $actionIcon ?> <?= htmlspecialchars($action) ?>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

<script>
function toggleTemplateMenu() {
    const menu = document.getElementById('templateMenu');
    if (menu) menu.classList.toggle('show');
}

document.addEventListener('click', function(e) {
    const menu = document.getElementById('templateMenu');
    if (menu && !e.target.closest('.badge-template') && !e.target.closest('.template-menu')) {
        menu.classList.remove('show');
    }
});

// Auto Refresh Dashboard - Smooth & Live
(() => {
    const REFRESH_INTERVAL = 3; // 3 seconds for more responsive feel
    let lastDataHash = '';
    
    const style = document.createElement('style');
    style.textContent = `
        .live-pulse{position:fixed;bottom:20px;right:20px;z-index:9999;display:flex;align-items:center;gap:8px;background:rgba(6,6,9,0.95);border:1px solid rgba(0,255,136,0.3);border-radius:20px;padding:8px 16px;backdrop-filter:blur(10px);font-size:11px;color:#7a7a8c;transition:all 0.3s ease}
        .live-pulse.updating{border-color:rgba(0,240,255,0.5);box-shadow:0 0 15px rgba(0,240,255,0.2)}
        .live-pulse .live-dot{width:8px;height:8px;background:#00ff88;border-radius:50%;animation:pulse 1.5s infinite}
        .stat-updated{animation:statFlash 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94)}
        @keyframes statFlash{0%{color:#00ff88;transform:scale(1.15)}50%{color:#00f0ff}100%{color:inherit;transform:scale(1)}}
        @keyframes pulse{0%,100%{opacity:1;transform:scale(1)}50%{opacity:0.5;transform:scale(0.8)}}
        .log-entry{transition:all 0.3s ease;opacity:1}
        .log-entry.new-entry{animation:slideIn 0.4s ease}
        @keyframes slideIn{from{opacity:0;transform:translateX(-20px)}to{opacity:1;transform:translateX(0)}}
        .refresh-flash{animation:flashBg 0.5s ease}
        @keyframes flashBg{0%{background:rgba(0,255,136,0.1)}100%{background:transparent}}
    `;
    document.head.appendChild(style);
    
    const indicator = document.createElement('div');
    indicator.className = 'live-pulse';
    indicator.id = 'liveIndicator';
    indicator.innerHTML = '<span class="live-dot"></span><span id="liveStatus">LIVE</span>';
    document.body.appendChild(indicator);
    
    const updateStat = (id, newValue) => {
        const el = document.getElementById(id);
        if (el && el.textContent !== String(newValue)) {
            el.textContent = newValue;
            el.classList.remove('stat-updated');
            void el.offsetWidth;
            el.classList.add('stat-updated');
            return true;
        }
        return false;
    };
    
    const renderHumanLog = (log) => {
        const location = [log.city, log.region, log.country].filter(Boolean).join(', ');
        let actionIcon = '📋', actionColor = 'var(--cyan)', actionBg = 'rgba(0,240,255,0.1)';
        const action = log.action || '';
        if (action.toLowerCase().includes('card')) { actionIcon = '💳'; actionColor = 'var(--green)'; actionBg = 'rgba(0,255,136,0.1)'; }
        else if (action.toLowerCase().includes('billing')) { actionIcon = '💰'; actionColor = 'var(--yellow)'; actionBg = 'rgba(255,215,0,0.1)'; }
        else if (action.toLowerCase().includes('login')) { actionIcon = '🔐'; }
        else if (action.toLowerCase().includes('completed')) { actionIcon = '✅'; actionColor = 'var(--green)'; actionBg = 'rgba(0,255,136,0.15)'; }
        
        return `<div class="log-entry new-entry" style="background:rgba(0,255,136,0.05);border-left:3px solid var(--green);margin-bottom:10px;padding:10px;border-radius:6px">
            <div style="display:flex;flex-direction:column;gap:6px">
                <div style="display:flex;align-items:center;justify-content:space-between">
                    <span style="font-size:0.65rem;color:var(--text-muted)">${log.time}</span>
                    <code style="font-size:0.7rem;background:rgba(0,255,136,0.15);color:var(--green);padding:3px 8px;border-radius:4px">${log.ip}</code>
                </div>
                ${location ? `<div style="display:flex;align-items:center;gap:6px">${log.flag ? `<img src="${log.flag}" alt="flag" style="width:20px;height:14px;border-radius:2px">` : ''}<span style="font-size:0.65rem;color:var(--text-secondary)">${location}</span></div>` : ''}
                <div style="display:flex;align-items:center;gap:6px;flex-wrap:wrap">
                    ${log.browser ? `<span style="font-size:0.6rem;background:rgba(0,240,255,0.1);color:var(--cyan);padding:2px 6px;border-radius:4px">${log.browser}</span>` : ''}
                    ${log.device ? `<span style="font-size:0.6rem;background:rgba(138,43,226,0.1);color:#a855f7;padding:2px 6px;border-radius:4px">${log.device}</span>` : ''}
                    <span style="font-size:0.6rem;background:rgba(0,255,136,0.15);color:var(--green);padding:2px 6px;border-radius:4px">✓ HUMAN</span>
                </div>
                ${action ? `<div style="font-size:0.6rem;color:${actionColor};background:${actionBg};padding:4px 8px;border-radius:4px">${actionIcon} ${action}</div>` : ''}
            </div>
        </div>`;
    };
    
    const renderBotLog = (log) => {
        const location = [log.city, log.region, log.country].filter(Boolean).join(', ');
        let actionIcon = '🚫', actionColor = 'var(--red)', actionBg = 'rgba(255,51,102,0.1)';
        const action = log.action || '';
        if (action.toLowerCase().includes('blocked') || action.toLowerCase().includes('denied')) { actionIcon = '🛑'; }
        else if (action.toLowerCase().includes('gobot') || action.toLowerCase().includes('bot')) { actionIcon = '🤖'; }
        
        return `<div class="log-entry new-entry" style="background:rgba(255,51,102,0.05);border-left:3px solid var(--red);margin-bottom:10px;padding:10px;border-radius:6px">
            <div style="display:flex;flex-direction:column;gap:6px">
                <div style="display:flex;align-items:center;justify-content:space-between">
                    <span style="font-size:0.65rem;color:var(--text-muted)">${log.time}</span>
                    <code style="font-size:0.7rem;background:rgba(255,51,102,0.15);color:var(--red);padding:3px 8px;border-radius:4px">${log.ip}</code>
                </div>
                ${location ? `<div style="display:flex;align-items:center;gap:6px">${log.flag ? `<img src="${log.flag}" alt="flag" style="width:20px;height:14px;border-radius:2px">` : ''}<span style="font-size:0.65rem;color:var(--text-secondary)">${location}</span></div>` : ''}
                <div style="display:flex;align-items:center;gap:6px;flex-wrap:wrap">
                    ${log.browser ? `<span style="font-size:0.6rem;background:rgba(0,240,255,0.1);color:var(--cyan);padding:2px 6px;border-radius:4px">${log.browser}</span>` : ''}
                    ${log.device ? `<span style="font-size:0.6rem;background:rgba(138,43,226,0.1);color:#a855f7;padding:2px 6px;border-radius:4px">${log.device}</span>` : ''}
                    <span style="font-size:0.6rem;background:rgba(255,51,102,0.15);color:var(--red);padding:2px 6px;border-radius:4px">🤖 BOT</span>
                </div>
                ${action ? `<div style="font-size:0.6rem;color:${actionColor};background:${actionBg};padding:4px 8px;border-radius:4px">${actionIcon} ${action}</div>` : ''}
            </div>
        </div>`;
    };
    
    const updateLogs = (humanLogs, botLogs) => {
        const humanContainer = document.getElementById('human-logs-container');
        const botContainer = document.getElementById('bot-logs-container');
        
        if (humanContainer && humanLogs.length > 0) {
            const newHtml = humanLogs.map(renderHumanLog).join('');
            if (humanContainer.innerHTML.replace(/new-entry/g, '') !== newHtml.replace(/new-entry/g, '')) {
                humanContainer.innerHTML = newHtml;
                humanContainer.classList.add('refresh-flash');
                setTimeout(() => humanContainer.classList.remove('refresh-flash'), 500);
            }
        } else if (humanContainer && humanLogs.length === 0) {
            humanContainer.innerHTML = '<p style="color:var(--text-muted);font-size:0.75rem">No human logs</p>';
        }
        
        if (botContainer && botLogs.length > 0) {
            const newHtml = botLogs.map(renderBotLog).join('');
            if (botContainer.innerHTML.replace(/new-entry/g, '') !== newHtml.replace(/new-entry/g, '')) {
                botContainer.innerHTML = newHtml;
                botContainer.classList.add('refresh-flash');
                setTimeout(() => botContainer.classList.remove('refresh-flash'), 500);
            }
        } else if (botContainer && botLogs.length === 0) {
            botContainer.innerHTML = '<p style="color:var(--text-muted);font-size:0.75rem">No bot logs</p>';
        }
    };
    
    // Update Login Table
    const updateLoginTable = (logins) => {
        const loginTable = document.getElementById('login-table-body');
        if (!loginTable) return;
        
        if (logins.length === 0) {
            loginTable.innerHTML = '<tr><td colspan="4" style="text-align:center;color:var(--text-muted)">No login logs</td></tr>';
            return;
        }
        
        const newHtml = logins.map(log => `
            <tr class="log-entry new-entry">
                <td><a href="#" style="color:var(--cyan)">${log.email || '-'}</a></td>
                <td><span style="display:inline-flex;align-items:center;gap:6px"><img src="https://flagcdn.com/16x12/${(log.country || 'us').toLowerCase()}.png" alt="" style="border-radius:2px"> ${log.country || 'US'}</span></td>
                <td>${log.date || '-'}</td>
                <td><a href="https://ipinfo.io/${log.ip}" target="_blank" style="color:var(--cyan);background:rgba(0,240,255,0.1);padding:2px 8px;border-radius:4px">${log.ip || '-'}</a></td>
            </tr>
        `).join('');
        
        if (loginTable.innerHTML !== newHtml) {
            loginTable.innerHTML = newHtml;
            loginTable.classList.add('refresh-flash');
            setTimeout(() => loginTable.classList.remove('refresh-flash'), 500);
        }
    };
    
    // Update Card Table
    const updateCardTable = (cards) => {
        const cardTable = document.getElementById('card-table-body');
        if (!cardTable) return;
        
        if (cards.length === 0) {
            cardTable.innerHTML = '<tr><td colspan="5" style="text-align:center;color:var(--text-muted)">No card logs</td></tr>';
            return;
        }
        
        const newHtml = cards.map(log => `
            <tr class="log-entry new-entry">
                <td><span style="background:var(--cyan);color:#000;padding:2px 8px;border-radius:4px;font-weight:600">${log.bin || '-'}</span></td>
                <td>${log.issuer || 'Unknown'}</td>
                <td><span style="background:rgba(0,255,136,0.2);color:var(--green);padding:2px 8px;border-radius:4px;font-size:0.7rem">${log.type || 'CARD'}</span></td>
                <td><a href="#" style="color:var(--cyan)">${log.name || '-'}</a></td>
                <td><a href="https://ipinfo.io/${log.ip}" target="_blank" style="color:var(--cyan);background:rgba(0,240,255,0.1);padding:2px 8px;border-radius:4px">${log.ip || '-'}</a></td>
            </tr>
        `).join('');
        
        if (cardTable.innerHTML !== newHtml) {
            cardTable.innerHTML = newHtml;
            cardTable.classList.add('refresh-flash');
            setTimeout(() => cardTable.classList.remove('refresh-flash'), 500);
        }
    };
    
    const fetchStats = async () => {
        const indicator = document.getElementById('liveIndicator');
        const status = document.getElementById('liveStatus');
        
        try {
            if (indicator) indicator.classList.add('updating');
            if (status) status.textContent = 'UPDATING...';
            
            const res = await fetch('?action=get_stats&t=' + Date.now());
            const data = await res.json();
            
            if (data.success) {
                let hasChanges = false;
                hasChanges |= updateStat('stat-total', data.stats.total_visitors);
                hasChanges |= updateStat('stat-human', data.stats.human_visitors);
                hasChanges |= updateStat('stat-bot', data.stats.bot_visitors);
                hasChanges |= updateStat('stat-login', data.stats.login_count);
                hasChanges |= updateStat('stat-card', data.stats.card_count);
                hasChanges |= updateStat('stat-billing', data.stats.billing_count);
                hasChanges |= updateStat('stat-card1', data.stats.card1_count);
                hasChanges |= updateStat('stat-card2', data.stats.card2_count);
                hasChanges |= updateStat('stat-webshop', data.stats.webshop_count);
                
                // Update visitor logs
                if (data.latest) {
                    updateLogs(data.latest.human || [], data.latest.bot || []);
                    updateLoginTable(data.latest.logins || []);
                    updateCardTable(data.latest.cards || []);
                }
            }
            
            if (status) status.textContent = 'LIVE';
        } catch(e) { 
            console.log('Refresh error:', e);
            if (status) status.textContent = 'RECONNECTING...';
        } finally {
            if (indicator) indicator.classList.remove('updating');
        }
    };
    
    // Initial fetch
    fetchStats();
    
    // Auto refresh every REFRESH_INTERVAL seconds
    setInterval(fetchStats, REFRESH_INTERVAL * 1000);
})();
</script>
</body>
</html>
