<?php
/**
 * ISP Whitelist Check
 * Checks if visitor's ISP is in the whitelist (config/isp_whitelist.json)
 * If whitelisted, visitor will NOT be blocked by antibot.
 * 
 * Usage: if (isIspWhitelisted($ip)) return 'no';
 */

if (!function_exists('isIspWhitelisted')) {
    function isIspWhitelisted($ip) {
        $wlFile = __DIR__ . '/../config/isp_whitelist.json';
        if (!file_exists($wlFile)) return false;
        
        $data = @json_decode(file_get_contents($wlFile), true);
        if (empty($data) || !is_array($data)) return false;
        
        // Get ISP for this IP (cached 1 hour)
        $isp = _getIspForIp($ip);
        if (empty($isp)) return false;
        
        // Partial match, case-insensitive
        $ispLower = strtolower($isp);
        foreach ($data as $entry) {
            $wl = strtolower(trim(is_array($entry) ? ($entry['isp'] ?? '') : $entry));
            if ($wl !== '' && strpos($ispLower, $wl) !== false) {
                return true;
            }
        }
        
        return false;
    }
}

if (!function_exists('_getIspForIp')) {
    function _getIspForIp($ip) {
        $configRoot = __DIR__ . '/../config/';
        // Check cache first (24 hour TTL)
        $cacheDir = sys_get_temp_dir();
        $cacheFile = $cacheDir . '/isp_wl_' . md5($ip) . '.txt';
        
        if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < 86400) {
            $cached = file_get_contents($cacheFile);
            if ($cached !== false && $cached !== '') {
                return $cached;
            }
        }
        
        // Get IPWhois key from gen.json
        $ipwhoisKey = 'cF8oCNZgptX8zTTj';
        $genFile = $configRoot . 'gen.json';
        if (file_exists($genFile)) {
            $genConfig = @json_decode(file_get_contents($genFile), true);
            if (!empty($genConfig['ipwhois_key'])) {
                $ipwhoisKey = $genConfig['ipwhois_key'];
            }
        }
        
        // Call IPWhois API
        $ch = curl_init("https://ipwhois.pro/{$ip}?key={$ipwhoisKey}");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 3);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $response = curl_exec($ch);
        curl_close($ch);
        
        $isp = '';
        if ($response) {
            $data = @json_decode($response, true);
            if (is_array($data)) {
                // IPWhois returns ISP in different fields depending on the endpoint
                $isp = $data['isp'] ?? $data['connection']['isp'] ?? '';
            }
        }
        
        // Cache the result
        if (!empty($isp)) {
            @file_put_contents($cacheFile, $isp);
        }
        
        return $isp;
    }
}
