<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// PHPMailer - Include inline (no composer needed)
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

// Auto-download PHPMailer if not exists
$phpmailerDir = __DIR__ . '/phpmailer';
if (!file_exists($phpmailerDir)) {
    mkdir($phpmailerDir, 0755, true);
}

$phpmailerFiles = [
    'PHPMailer.php' => 'https://raw.githubusercontent.com/PHPMailer/PHPMailer/master/src/PHPMailer.php',
    'SMTP.php' => 'https://raw.githubusercontent.com/PHPMailer/PHPMailer/master/src/SMTP.php',
    'Exception.php' => 'https://raw.githubusercontent.com/PHPMailer/PHPMailer/master/src/Exception.php'
];

foreach ($phpmailerFiles as $file => $url) {
    $filePath = $phpmailerDir . '/' . $file;
    if (!file_exists($filePath)) {
        $content = @file_get_contents($url);
        if ($content) {
            file_put_contents($filePath, $content);
        }
    }
}

// Include PHPMailer
if (file_exists($phpmailerDir . '/Exception.php')) {
    require $phpmailerDir . '/Exception.php';
    require $phpmailerDir . '/PHPMailer.php';
    require $phpmailerDir . '/SMTP.php';
}

// Debug log file
$debugLogFile = __DIR__ . '/mail-debug.log';

function writeDebugLog($message, $data = null) {
    global $debugLogFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message";
    if ($data !== null) {
        $logEntry .= "\n" . print_r($data, true);
    }
    $logEntry .= "\n" . str_repeat('-', 50) . "\n";
    file_put_contents($debugLogFile, $logEntry, FILE_APPEND);
}

// Function to read full SMTP response (handles multi-line)
function smtpReadResponse($socket, &$debugOutput) {
    $response = '';
    $lines = [];
    
    while (true) {
        $line = fgets($socket, 515);
        if ($line === false) break;
        
        $lines[] = trim($line);
        $response .= $line;
        
        // Check if this is the last line (format: "XXX " not "XXX-")
        if (isset($line[3]) && $line[3] === ' ') {
            break;
        }
        // Also break if line starts with 3 digits followed by space
        if (preg_match('/^\d{3} /', $line)) {
            break;
        }
    }
    
    foreach ($lines as $l) {
        $debugOutput[] = "  >> " . $l;
    }
    
    return ['full' => $response, 'lines' => $lines, 'code' => (int)substr($response, 0, 3)];
}

// Function to send SMTP command and get response
function smtpCommand($socket, $command, &$debugOutput) {
    $debugOutput[] = "<< " . trim($command);
    fputs($socket, $command . "\r\n");
    return smtpReadResponse($socket, $debugOutput);
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    $debugOutput = [];
    $debugOutput[] = "=== DEBUG START ===";
    $debugOutput[] = "Time: " . date('Y-m-d H:i:s');
    $debugOutput[] = "Action: " . $_POST['action'];
    
    // SMTP Relay (Raw Socket)
    if ($_POST['action'] === 'smtp_relay') {
        $host = $_POST['smtp_host'] ?? 'localhost';
        $port = (int)($_POST['smtp_port'] ?? 25);
        $from_email = $_POST['from_email'] ?? '';
        $from_name = $_POST['from_name'] ?? '';
        $send_to = $_POST['send_to'] ?? '';
        $subject = $_POST['subject'] ?? 'Test Email from Mail Sender';
        $message = $_POST['message'] ?? '<h1>Test Email</h1><p>This is a test email sent via SMTP Relay.</p>';
        
        $debugOutput[] = "Host: $host";
        $debugOutput[] = "Port: $port";
        $debugOutput[] = "From: $from_name <$from_email>";
        $debugOutput[] = "To: $send_to";
        
        writeDebugLog("SMTP Relay Request", $_POST);
        
        if (empty($from_email) || empty($send_to)) {
            echo json_encode(['success' => false, 'message' => 'From Email dan Send To harus diisi!', 'debug' => $debugOutput]);
            exit;
        }
        
        try {
            $debugOutput[] = "Connecting to $host:$port...";
            $socket = @fsockopen($host, $port, $errno, $errstr, 30);
            
            if (!$socket) {
                $debugOutput[] = "ERROR: Connection failed - $errstr ($errno)";
                writeDebugLog("SMTP Connection Failed", ['error' => $errstr, 'errno' => $errno]);
                echo json_encode(['success' => false, 'message' => "SMTP Connection failed: $errstr ($errno)", 'debug' => $debugOutput]);
                exit;
            }
            
            stream_set_timeout($socket, 30);
            $debugOutput[] = "Connected successfully!";
            
            // Read server greeting
            $debugOutput[] = "[SERVER GREETING]";
            $greeting = smtpReadResponse($socket, $debugOutput);
            if ($greeting['code'] != 220) {
                fclose($socket);
                echo json_encode(['success' => false, 'message' => "Server rejected connection: " . $greeting['full'], 'debug' => $debugOutput]);
                exit;
            }
            
            // Send EHLO
            $debugOutput[] = "[EHLO]";
            $ehlo = smtpCommand($socket, "EHLO localhost", $debugOutput);
            if ($ehlo['code'] != 250) {
                // Try HELO if EHLO fails
                $debugOutput[] = "[HELO fallback]";
                $helo = smtpCommand($socket, "HELO localhost", $debugOutput);
                if ($helo['code'] != 250) {
                    fclose($socket);
                    echo json_encode(['success' => false, 'message' => "HELO failed: " . $helo['full'], 'debug' => $debugOutput]);
                    exit;
                }
            }
            
            // MAIL FROM
            $debugOutput[] = "[MAIL FROM]";
            $mailFrom = smtpCommand($socket, "MAIL FROM: <$from_email>", $debugOutput);
            if ($mailFrom['code'] != 250) {
                fclose($socket);
                echo json_encode(['success' => false, 'message' => "MAIL FROM rejected: " . $mailFrom['full'], 'debug' => $debugOutput]);
                exit;
            }
            
            // RCPT TO
            $debugOutput[] = "[RCPT TO]";
            $rcptTo = smtpCommand($socket, "RCPT TO: <$send_to>", $debugOutput);
            if ($rcptTo['code'] != 250 && $rcptTo['code'] != 251) {
                fclose($socket);
                echo json_encode(['success' => false, 'message' => "RCPT TO rejected: " . $rcptTo['full'], 'debug' => $debugOutput]);
                exit;
            }
            
            // DATA
            $debugOutput[] = "[DATA]";
            $data = smtpCommand($socket, "DATA", $debugOutput);
            if ($data['code'] != 354) {
                fclose($socket);
                echo json_encode(['success' => false, 'message' => "DATA rejected: " . $data['full'], 'debug' => $debugOutput]);
                exit;
            }
            
            // Build email
            $headers = "From: $from_name <$from_email>\r\n";
            $headers .= "To: <$send_to>\r\n";
            $headers .= "Subject: $subject\r\n";
            $headers .= "Date: " . date('r') . "\r\n";
            $headers .= "Message-ID: <" . uniqid() . "@" . gethostname() . ">\r\n";
            $headers .= "MIME-Version: 1.0\r\n";
            $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
            $headers .= "X-Mailer: PHP Mail Sender Debug\r\n";
            
            $emailContent = $headers . "\r\n" . $message . "\r\n.\r\n";
            
            $debugOutput[] = "[SENDING MESSAGE]";
            $debugOutput[] = "<< [Email Headers + Body]";
            fputs($socket, $emailContent);
            
            $msgResponse = smtpReadResponse($socket, $debugOutput);
            
            // Check if message was accepted (250 OK)
            if ($msgResponse['code'] != 250) {
                fclose($socket);
                echo json_encode(['success' => false, 'message' => "Message rejected: " . $msgResponse['full'], 'debug' => $debugOutput]);
                exit;
            }
            
            // QUIT
            $debugOutput[] = "[QUIT]";
            smtpCommand($socket, "QUIT", $debugOutput);
            fclose($socket);
            
            $debugOutput[] = "=== EMAIL SENT SUCCESSFULLY ===";
            $debugOutput[] = "=== DEBUG END ===";
            writeDebugLog("SMTP Relay Success", $debugOutput);
            
            echo json_encode(['success' => true, 'message' => 'Email berhasil dikirim via SMTP Relay!', 'debug' => $debugOutput]);
        } catch (Exception $e) {
            $debugOutput[] = "ERROR: " . $e->getMessage();
            writeDebugLog("SMTP Relay Error", $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage(), 'debug' => $debugOutput]);
        }
        exit;
    }
    
    // PHP mail()
    if ($_POST['action'] === 'php_mail') {
        $from_email = $_POST['from_email'] ?? '';
        $from_name = $_POST['from_name'] ?? '';
        $send_to = $_POST['send_to'] ?? '';
        $subject = $_POST['subject'] ?? 'Test Email from Mail Sender';
        $message = $_POST['message'] ?? '<h1>Test Email</h1><p>This is a test email sent via PHP mail().</p>';
        
        $debugOutput[] = "Method: PHP mail()";
        $debugOutput[] = "From: $from_name <$from_email>";
        $debugOutput[] = "To: $send_to";
        $debugOutput[] = "Subject: $subject";
        
        writeDebugLog("PHP mail() Request", $_POST);
        
        if (empty($from_email) || empty($send_to)) {
            echo json_encode(['success' => false, 'message' => 'From Email dan Send To harus diisi!', 'debug' => $debugOutput]);
            exit;
        }
        
        $headers = "From: $from_name <$from_email>\r\n";
        $headers .= "Reply-To: $from_email\r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "X-Mailer: PHP Mail Sender Debug\r\n";
        
        $debugOutput[] = "Headers: " . str_replace("\r\n", " | ", $headers);
        
        if (@mail($send_to, $subject, $message, $headers)) {
            $debugOutput[] = "mail() returned TRUE";
            $debugOutput[] = "=== DEBUG END ===";
            writeDebugLog("PHP mail() Success", $debugOutput);
            echo json_encode(['success' => true, 'message' => 'Email berhasil dikirim via PHP mail()!', 'debug' => $debugOutput]);
        } else {
            $debugOutput[] = "mail() returned FALSE";
            $error = error_get_last();
            if ($error) {
                $debugOutput[] = "Last Error: " . $error['message'];
            }
            $debugOutput[] = "=== DEBUG END ===";
            writeDebugLog("PHP mail() Failed", $debugOutput);
            echo json_encode(['success' => false, 'message' => 'Gagal mengirim email via PHP mail()', 'debug' => $debugOutput]);
        }
        exit;
    }
    
    // PHPMailer SMTP
    if ($_POST['action'] === 'phpmailer') {
        $smtp_host = $_POST['smtp_host'] ?? '';
        $smtp_port = (int)($_POST['smtp_port'] ?? 587);
        $smtp_user = $_POST['smtp_user'] ?? '';
        $smtp_pass = $_POST['smtp_pass'] ?? '';
        $smtp_secure = $_POST['smtp_secure'] ?? 'tls';
        $from_email = $_POST['from_email'] ?? '';
        $from_name = $_POST['from_name'] ?? '';
        $send_to = $_POST['send_to'] ?? '';
        $subject = $_POST['subject'] ?? 'Test Email from PHPMailer';
        $message = $_POST['message'] ?? '<h1>Test Email</h1><p>This is a test email sent via PHPMailer.</p>';
        
        $debugOutput[] = "Method: PHPMailer SMTP";
        $debugOutput[] = "SMTP Host: $smtp_host";
        $debugOutput[] = "SMTP Port: $smtp_port";
        $debugOutput[] = "SMTP User: $smtp_user";
        $debugOutput[] = "SMTP Secure: $smtp_secure";
        $debugOutput[] = "From: $from_name <$from_email>";
        $debugOutput[] = "To: $send_to";
        
        writeDebugLog("PHPMailer Request", $_POST);
        
        if (empty($smtp_host) || empty($from_email) || empty($send_to)) {
            echo json_encode(['success' => false, 'message' => 'SMTP Host, From Email, dan Send To harus diisi!', 'debug' => $debugOutput]);
            exit;
        }
        
        if (!class_exists('PHPMailer\PHPMailer\PHPMailer')) {
            echo json_encode(['success' => false, 'message' => 'PHPMailer tidak tersedia. Silakan refresh halaman untuk download otomatis.', 'debug' => $debugOutput]);
            exit;
        }
        
        try {
            $mail = new PHPMailer(true);
            
            // Enable debug
            $mail->SMTPDebug = SMTP::DEBUG_SERVER;
            $mail->Debugoutput = function($str, $level) use (&$debugOutput) {
                $debugOutput[] = "SMTP[$level]: " . trim($str);
            };
            
            // Server settings
            $mail->isSMTP();
            $mail->Host = $smtp_host;
            $mail->SMTPAuth = !empty($smtp_user);
            if (!empty($smtp_user)) {
                $mail->Username = $smtp_user;
                $mail->Password = $smtp_pass;
            }
            
            if ($smtp_secure === 'tls') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            } elseif ($smtp_secure === 'ssl') {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            } else {
                $mail->SMTPSecure = '';
                $mail->SMTPAutoTLS = false;
            }
            
            $mail->Port = $smtp_port;
            $mail->CharSet = 'UTF-8';
            
            // Timeout
            $mail->Timeout = 30;
            
            // Recipients
            $mail->setFrom($from_email, $from_name);
            $mail->addAddress($send_to);
            $mail->addReplyTo($from_email, $from_name);
            
            // Content
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $message;
            $mail->AltBody = strip_tags($message);
            
            $debugOutput[] = "Attempting to send...";
            
            $mail->send();
            
            $debugOutput[] = "Email sent successfully!";
            $debugOutput[] = "=== DEBUG END ===";
            writeDebugLog("PHPMailer Success", $debugOutput);
            
            echo json_encode(['success' => true, 'message' => 'Email berhasil dikirim via PHPMailer!', 'debug' => $debugOutput]);
        } catch (Exception $e) {
            $debugOutput[] = "ERROR: " . $mail->ErrorInfo;
            $debugOutput[] = "=== DEBUG END ===";
            writeDebugLog("PHPMailer Error", ['error' => $mail->ErrorInfo, 'debug' => $debugOutput]);
            echo json_encode(['success' => false, 'message' => 'PHPMailer Error: ' . $mail->ErrorInfo, 'debug' => $debugOutput]);
        }
        exit;
    }
    
    // Clear debug log
    if ($_POST['action'] === 'clear_log') {
        if (file_exists($debugLogFile)) {
            unlink($debugLogFile);
        }
        echo json_encode(['success' => true, 'message' => 'Debug log cleared!']);
        exit;
    }
    
    // Get debug log
    if ($_POST['action'] === 'get_log') {
        $log = file_exists($debugLogFile) ? file_get_contents($debugLogFile) : 'No log entries yet.';
        echo json_encode(['success' => true, 'log' => $log]);
        exit;
    }
}

// Check PHPMailer status
$phpmailerStatus = class_exists('PHPMailer\PHPMailer\PHPMailer') ? 'available' : 'not_available';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mail Sender - Debug Mode</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&family=JetBrains+Mono:wght@400;500&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: #0a0a0a;
            min-height: 100vh;
            color: #fff;
            padding: 20px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            flex-wrap: wrap;
            gap: 15px;
        }
        
        .mode-selector {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .mode-icon {
            font-size: 24px;
        }
        
        .mode-label {
            font-size: 14px;
            color: #888;
        }
        
        .mode-btn {
            padding: 10px 20px;
            border-radius: 25px;
            border: none;
            cursor: pointer;
            font-weight: 600;
            font-size: 13px;
            transition: all 0.3s ease;
        }
        
        .mode-btn.active {
            background: linear-gradient(135deg, #00d4aa, #00b894);
            color: #000;
        }
        
        .mode-btn:not(.active) {
            background: transparent;
            color: #888;
            border: 1px solid #333;
        }
        
        .mode-btn:hover:not(.active) {
            border-color: #00d4aa;
            color: #00d4aa;
        }
        
        .toggle-btns {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .toggle-btn {
            padding: 10px 20px;
            border-radius: 25px;
            border: 1px solid #333;
            background: transparent;
            color: #888;
            cursor: pointer;
            font-size: 13px;
            transition: all 0.3s ease;
        }
        
        .toggle-btn.active {
            background: linear-gradient(135deg, #00d4aa, #00b894);
            color: #000;
            border: none;
        }
        
        .toggle-btn:hover:not(.active) {
            border-color: #00d4aa;
            color: #00d4aa;
        }
        
        .container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 25px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .card {
            background: #111;
            border-radius: 16px;
            padding: 25px;
            border: 2px solid #222;
            transition: all 0.3s ease;
        }
        
        .card.active {
            border-color: #00d4aa;
            box-shadow: 0 0 30px rgba(0, 212, 170, 0.1);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .card-title {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .card-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
        }
        
        .smtp-icon {
            background: linear-gradient(135deg, #ffd700, #ffaa00);
        }
        
        .mail-icon {
            background: linear-gradient(135deg, #00d4aa, #00b894);
        }
        
        .phpmailer-icon {
            background: linear-gradient(135deg, #667eea, #764ba2);
        }
        
        .card-title h3 {
            font-size: 18px;
            font-weight: 600;
        }
        
        .badge {
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 11px;
            font-weight: 600;
            margin-left: 10px;
        }
        
        .badge-warning {
            background: rgba(255, 215, 0, 0.2);
            color: #ffd700;
        }
        
        .badge-info {
            background: rgba(0, 212, 170, 0.2);
            color: #00d4aa;
        }
        
        .badge-purple {
            background: rgba(102, 126, 234, 0.2);
            color: #667eea;
        }
        
        .badge-success {
            background: rgba(0, 212, 170, 0.2);
            color: #00d4aa;
        }
        
        .badge-danger {
            background: rgba(255, 82, 82, 0.2);
            color: #ff5252;
        }
        
        .status-badge {
            padding: 6px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-use {
            background: transparent;
            color: #666;
            border: 1px solid #333;
        }
        
        .status-active {
            background: rgba(0, 212, 170, 0.2);
            color: #00d4aa;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .form-row.triple {
            grid-template-columns: 1fr 1fr 1fr;
        }
        
        .form-row.full {
            grid-template-columns: 1fr;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-group label {
            display: block;
            font-size: 12px;
            color: #666;
            margin-bottom: 8px;
            font-weight: 500;
        }
        
        .form-group label span {
            color: #00d4aa;
        }
        
        .form-control {
            width: 100%;
            padding: 14px 16px;
            background: #1a1a1a;
            border: 1px solid #333;
            border-radius: 10px;
            color: #fff;
            font-size: 14px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #00d4aa;
            box-shadow: 0 0 0 3px rgba(0, 212, 170, 0.1);
        }
        
        .form-control::placeholder {
            color: #555;
        }
        
        select.form-control {
            cursor: pointer;
        }
        
        textarea.form-control {
            min-height: 100px;
            resize: vertical;
            font-family: 'JetBrains Mono', monospace;
            font-size: 13px;
        }
        
        .btn {
            width: 100%;
            padding: 16px;
            border: none;
            border-radius: 12px;
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            transition: all 0.3s ease;
            margin-top: 10px;
        }
        
        .btn-smtp {
            background: linear-gradient(135deg, #ffd700, #ffaa00);
            color: #000;
        }
        
        .btn-smtp:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(255, 215, 0, 0.3);
        }
        
        .btn-mail {
            background: linear-gradient(135deg, #00d4aa, #00b894);
            color: #000;
        }
        
        .btn-mail:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(0, 212, 170, 0.3);
        }
        
        .btn-phpmailer {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: #fff;
        }
        
        .btn-phpmailer:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
        }
        
        .btn-debug {
            background: linear-gradient(135deg, #ff6b6b, #ee5a5a);
            color: #fff;
        }
        
        .btn-debug:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px rgba(255, 107, 107, 0.3);
        }
        
        .btn-secondary {
            background: #222;
            color: #888;
        }
        
        .btn-secondary:hover {
            background: #333;
            color: #fff;
        }
        
        .alert {
            padding: 14px 18px;
            border-radius: 10px;
            margin-top: 15px;
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 14px;
            display: none;
        }
        
        .alert-error {
            background: rgba(255, 82, 82, 0.15);
            border: 1px solid rgba(255, 82, 82, 0.3);
            color: #ff5252;
        }
        
        .alert-success {
            background: rgba(0, 212, 170, 0.15);
            border: 1px solid rgba(0, 212, 170, 0.3);
            color: #00d4aa;
        }
        
        /* Debug Console */
        .debug-panel {
            background: #111;
            border-radius: 16px;
            padding: 25px;
            border: 2px solid #333;
            margin-top: 25px;
        }
        
        .debug-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .debug-title {
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 18px;
            font-weight: 600;
        }
        
        .debug-title .icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            background: linear-gradient(135deg, #ff6b6b, #ee5a5a);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
        }
        
        .debug-actions {
            display: flex;
            gap: 10px;
        }
        
        .debug-btn {
            padding: 8px 16px;
            border-radius: 8px;
            border: 1px solid #333;
            background: transparent;
            color: #888;
            cursor: pointer;
            font-size: 12px;
            transition: all 0.3s ease;
        }
        
        .debug-btn:hover {
            border-color: #00d4aa;
            color: #00d4aa;
        }
        
        .debug-console {
            background: #0a0a0a;
            border-radius: 12px;
            padding: 20px;
            font-family: 'JetBrains Mono', monospace;
            font-size: 12px;
            line-height: 1.8;
            max-height: 400px;
            overflow-y: auto;
            white-space: pre-wrap;
            word-break: break-all;
        }
        
        .debug-line {
            padding: 4px 0;
            border-bottom: 1px solid #1a1a1a;
        }
        
        .debug-line.success {
            color: #00d4aa;
        }
        
        .debug-line.error {
            color: #ff5252;
        }
        
        .debug-line.info {
            color: #667eea;
        }
        
        .debug-line.warning {
            color: #ffd700;
        }
        
        .footer {
            margin-top: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            background: #111;
            border-radius: 12px;
            flex-wrap: wrap;
            gap: 15px;
        }
        
        .footer-info {
            display: flex;
            align-items: center;
            gap: 10px;
            color: #666;
            font-size: 13px;
        }
        
        .footer-info .icon {
            color: #00d4aa;
        }
        
        .footer-btn {
            padding: 10px 20px;
            background: transparent;
            border: 1px solid #333;
            border-radius: 8px;
            color: #888;
            font-size: 13px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .footer-btn:hover {
            border-color: #00d4aa;
            color: #00d4aa;
        }
        
        .divider {
            height: 1px;
            background: #222;
            margin: 20px 0;
        }
        
        @media (max-width: 900px) {
            .container {
                grid-template-columns: 1fr;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
            }
            
            .form-row {
                grid-template-columns: 1fr;
            }
            
            .form-row.triple {
                grid-template-columns: 1fr;
            }
        }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 2px solid transparent;
            border-top-color: currentColor;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Scrollbar */
        ::-webkit-scrollbar {
            width: 8px;
        }
        
        ::-webkit-scrollbar-track {
            background: #1a1a1a;
            border-radius: 4px;
        }
        
        ::-webkit-scrollbar-thumb {
            background: #333;
            border-radius: 4px;
        }
        
        ::-webkit-scrollbar-thumb:hover {
            background: #444;
        }
        
        .server-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .info-item {
            background: #1a1a1a;
            padding: 12px 16px;
            border-radius: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .info-label {
            color: #666;
            font-size: 12px;
        }
        
        .info-value {
            color: #00d4aa;
            font-size: 13px;
            font-weight: 600;
            font-family: 'JetBrains Mono', monospace;
        }
        
        .tabs {
            display: flex;
            gap: 5px;
            margin-bottom: 20px;
            background: #1a1a1a;
            padding: 5px;
            border-radius: 12px;
        }
        
        .tab {
            flex: 1;
            padding: 12px;
            text-align: center;
            border-radius: 8px;
            cursor: pointer;
            font-size: 13px;
            font-weight: 500;
            transition: all 0.3s ease;
            color: #666;
        }
        
        .tab.active {
            background: linear-gradient(135deg, #00d4aa, #00b894);
            color: #000;
        }
        
        .tab:hover:not(.active) {
            color: #fff;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="mode-selector">
            <span class="mode-icon">🔧</span>
            <span class="mode-label">Mail Sender</span>
            <span class="badge badge-<?php echo $phpmailerStatus === 'available' ? 'success' : 'danger'; ?>">
                PHPMailer: <?php echo $phpmailerStatus === 'available' ? 'Ready' : 'Not Available'; ?>
            </span>
        </div>
        <div class="toggle-btns">
            <button class="toggle-btn active" data-tab="smtp">⚡ SMTP Relay</button>
            <button class="toggle-btn" data-tab="phpmail">📧 PHP mail()</button>
            <button class="toggle-btn" data-tab="phpmailer">📦 PHPMailer</button>
        </div>
    </div>
    
    <!-- Server Info -->
    <div class="server-info">
        <div class="info-item">
            <span class="info-label">Server IP</span>
            <span class="info-value"><?php echo $_SERVER['SERVER_ADDR'] ?? 'N/A'; ?></span>
        </div>
        <div class="info-item">
            <span class="info-label">PHP Version</span>
            <span class="info-value"><?php echo phpversion(); ?></span>
        </div>
        <div class="info-item">
            <span class="info-label">Mail Function</span>
            <span class="info-value"><?php echo function_exists('mail') ? 'Enabled' : 'Disabled'; ?></span>
        </div>
        <div class="info-item">
            <span class="info-label">OpenSSL</span>
            <span class="info-value"><?php echo extension_loaded('openssl') ? 'Enabled' : 'Disabled'; ?></span>
        </div>
    </div>
    
    <div class="container">
        <!-- SMTP Relay Card -->
        <div class="tab-content active" id="tab-smtp">
            <div class="card active">
                <div class="card-header">
                    <div class="card-title">
                        <div class="card-icon smtp-icon">⚡</div>
                        <h3>SMTP Relay <span class="badge badge-warning">No Auth</span></h3>
                    </div>
                    <span class="status-badge status-active">● ACTIVE</span>
                </div>
                
                <form id="smtp-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label>SMTP Host</label>
                            <input type="text" class="form-control" name="smtp_host" value="localhost" placeholder="localhost">
                        </div>
                        <div class="form-group">
                            <label>Port</label>
                            <input type="text" class="form-control" name="smtp_port" value="25" placeholder="25">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>From Email <span>*</span></label>
                            <input type="email" class="form-control" name="from_email" placeholder="noreply@domain.com">
                        </div>
                        <div class="form-group">
                            <label>From Name</label>
                            <input type="text" class="form-control" name="from_name" value="Mail Sender" placeholder="Sender Name">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Send To <span>*</span></label>
                        <input type="email" class="form-control" name="send_to" placeholder="your-email@gmail.com">
                    </div>
                    
                    <div class="form-group">
                        <label>Subject</label>
                        <input type="text" class="form-control" name="subject" value="Test Email - SMTP Relay" placeholder="Email Subject">
                    </div>
                    
                    <div class="form-group">
                        <label>Message (HTML)</label>
                        <textarea class="form-control" name="message" placeholder="<h1>Hello!</h1><p>This is a test email.</p>"><h1>Test Email</h1><p>This email was sent via <strong>SMTP Relay</strong> method.</p><p>Timestamp: <?php echo date('Y-m-d H:i:s'); ?></p></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-smtp">
                        <span>⚡</span> Send via SMTP Relay
                    </button>
                    
                    <div class="alert alert-error" id="smtp-error">
                        <span>✕</span> <span class="message"></span>
                    </div>
                    <div class="alert alert-success" id="smtp-success">
                        <span>✓</span> <span class="message"></span>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- PHP mail() Card -->
        <div class="tab-content" id="tab-phpmail">
            <div class="card">
                <div class="card-header">
                    <div class="card-title">
                        <div class="card-icon mail-icon">📧</div>
                        <h3>PHP mail() <span class="badge badge-info">Sendmail</span></h3>
                    </div>
                    <span class="status-badge status-use">● USE THIS</span>
                </div>
                
                <form id="mail-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label>From Email <span>*</span></label>
                            <input type="email" class="form-control" name="from_email" placeholder="noreply@domain.com">
                        </div>
                        <div class="form-group">
                            <label>From Name</label>
                            <input type="text" class="form-control" name="from_name" value="Mail Sender" placeholder="Sender Name">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Send To <span>*</span></label>
                        <input type="email" class="form-control" name="send_to" placeholder="your-email@gmail.com">
                    </div>
                    
                    <div class="form-group">
                        <label>Subject</label>
                        <input type="text" class="form-control" name="subject" value="Test Email - PHP mail()" placeholder="Email Subject">
                    </div>
                    
                    <div class="form-group">
                        <label>Message (HTML)</label>
                        <textarea class="form-control" name="message" placeholder="<h1>Hello!</h1><p>This is a test email.</p>"><h1>Test Email</h1><p>This email was sent via <strong>PHP mail()</strong> function.</p><p>Timestamp: <?php echo date('Y-m-d H:i:s'); ?></p></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-mail">
                        <span>📧</span> Send via PHP mail()
                    </button>
                    
                    <div class="alert alert-error" id="mail-error">
                        <span>✕</span> <span class="message"></span>
                    </div>
                    <div class="alert alert-success" id="mail-success">
                        <span>✓</span> <span class="message"></span>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- PHPMailer Card -->
        <div class="tab-content" id="tab-phpmailer">
            <div class="card">
                <div class="card-header">
                    <div class="card-title">
                        <div class="card-icon phpmailer-icon">📦</div>
                        <h3>PHPMailer <span class="badge badge-purple">With Auth</span></h3>
                    </div>
                    <span class="status-badge status-use">● RECOMMENDED</span>
                </div>
                
                <form id="phpmailer-form">
                    <div class="form-row triple">
                        <div class="form-group">
                            <label>SMTP Host <span>*</span></label>
                            <input type="text" class="form-control" name="smtp_host" placeholder="smtp.gmail.com">
                        </div>
                        <div class="form-group">
                            <label>Port</label>
                            <input type="text" class="form-control" name="smtp_port" value="587" placeholder="587">
                        </div>
                        <div class="form-group">
                            <label>Security</label>
                            <select class="form-control" name="smtp_secure">
                                <option value="tls">TLS (587)</option>
                                <option value="ssl">SSL (465)</option>
                                <option value="none">None (25)</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>SMTP Username</label>
                            <input type="text" class="form-control" name="smtp_user" placeholder="your-email@gmail.com">
                        </div>
                        <div class="form-group">
                            <label>SMTP Password / App Password</label>
                            <input type="password" class="form-control" name="smtp_pass" placeholder="App Password (16 characters)">
                        </div>
                    </div>
                    
                    <div class="divider"></div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>From Email <span>*</span></label>
                            <input type="email" class="form-control" name="from_email" placeholder="your-email@gmail.com">
                        </div>
                        <div class="form-group">
                            <label>From Name</label>
                            <input type="text" class="form-control" name="from_name" value="Mail Sender" placeholder="Sender Name">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Send To <span>*</span></label>
                        <input type="email" class="form-control" name="send_to" placeholder="recipient@example.com">
                    </div>
                    
                    <div class="form-group">
                        <label>Subject</label>
                        <input type="text" class="form-control" name="subject" value="Test Email - PHPMailer" placeholder="Email Subject">
                    </div>
                    
                    <div class="form-group">
                        <label>Message (HTML)</label>
                        <textarea class="form-control" name="message" placeholder="<h1>Hello!</h1><p>This is a test email.</p>"><h1>Test Email</h1><p>This email was sent via <strong>PHPMailer</strong> with SMTP authentication.</p><p>Timestamp: <?php echo date('Y-m-d H:i:s'); ?></p></textarea>
                    </div>
                    
                    <button type="submit" class="btn btn-phpmailer">
                        <span>📦</span> Send via PHPMailer
                    </button>
                    
                    <div class="alert alert-error" id="phpmailer-error">
                        <span>✕</span> <span class="message"></span>
                    </div>
                    <div class="alert alert-success" id="phpmailer-success">
                        <span>✓</span> <span class="message"></span>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Debug Console -->
    <div class="debug-panel">
        <div class="debug-header">
            <div class="debug-title">
                <div class="icon">🐛</div>
                <span>Debug Console</span>
            </div>
            <div class="debug-actions">
                <button class="debug-btn" onclick="loadDebugLog()">📋 Load Log</button>
                <button class="debug-btn" onclick="clearDebugLog()">🗑️ Clear Log</button>
                <button class="debug-btn" onclick="clearConsole()">🧹 Clear Console</button>
            </div>
        </div>
        <div class="debug-console" id="debug-console">
<span class="debug-line info">=== Mail Sender Debug Console ===</span>
<span class="debug-line">Ready to send emails. Select a method and fill the form.</span>
<span class="debug-line warning">Tip: For Gmail, use App Password instead of your regular password.</span>
<span class="debug-line">Generate App Password: Google Account > Security > 2-Step Verification > App passwords</span>
        </div>
    </div>
    
    <div class="footer">
        <div class="footer-info">
            <span class="icon">ℹ</span>
            <strong>Tips:</strong> SMTP Relay (localhost:25) tidak perlu auth. PHPMailer dengan Gmail butuh App Password.
        </div>
        <button class="footer-btn" onclick="window.location.reload()">🔄 Refresh</button>
    </div>
    
    <script>
        const debugConsole = document.getElementById('debug-console');
        
        // Tab switching
        document.querySelectorAll('.toggle-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                document.querySelectorAll('.toggle-btn').forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                
                const tab = this.dataset.tab;
                document.querySelectorAll('.tab-content').forEach(t => t.classList.remove('active'));
                document.getElementById('tab-' + tab).classList.add('active');
            });
        });
        
        function addDebugLine(message, type = '') {
            const line = document.createElement('span');
            line.className = 'debug-line ' + type;
            line.textContent = message;
            debugConsole.appendChild(line);
            debugConsole.scrollTop = debugConsole.scrollHeight;
        }
        
        function clearConsole() {
            debugConsole.innerHTML = '<span class="debug-line info">=== Console Cleared ===</span>';
        }
        
        async function loadDebugLog() {
            const formData = new FormData();
            formData.append('action', 'get_log');
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                const result = await response.json();
                
                addDebugLine('=== Debug Log File ===', 'info');
                result.log.split('\n').forEach(line => {
                    if (line.trim()) addDebugLine(line);
                });
            } catch (error) {
                addDebugLine('Error loading log: ' + error.message, 'error');
            }
        }
        
        async function clearDebugLog() {
            const formData = new FormData();
            formData.append('action', 'clear_log');
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                const result = await response.json();
                addDebugLine(result.message, 'success');
            } catch (error) {
                addDebugLine('Error clearing log: ' + error.message, 'error');
            }
        }
        
        // SMTP Form
        document.getElementById('smtp-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            await sendMail(this, 'smtp_relay', 'smtp');
        });
        
        // PHP mail() Form
        document.getElementById('mail-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            await sendMail(this, 'php_mail', 'mail');
        });
        
        // PHPMailer Form
        document.getElementById('phpmailer-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            await sendMail(this, 'phpmailer', 'phpmailer');
        });
        
        async function sendMail(form, action, prefix) {
            const btn = form.querySelector('button[type="submit"]');
            const originalText = btn.innerHTML;
            btn.innerHTML = '<span class="loading"></span> Sending...';
            btn.disabled = true;
            
            hideAlerts(prefix);
            addDebugLine('=== Starting ' + action.toUpperCase() + ' ===', 'info');
            
            const formData = new FormData(form);
            formData.append('action', action);
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                // Show debug output
                if (result.debug) {
                    result.debug.forEach(line => {
                        let type = '';
                        if (line.includes('ERROR')) type = 'error';
                        else if (line.includes('success') || line.includes('Successfully')) type = 'success';
                        else if (line.includes('DEBUG')) type = 'info';
                        else if (line.includes('WARNING') || line.includes('Tip')) type = 'warning';
                        addDebugLine(line, type);
                    });
                }
                
                if (result.success) {
                    showAlert(prefix, 'success', result.message);
                    addDebugLine('✓ ' + result.message, 'success');
                } else {
                    showAlert(prefix, 'error', result.message);
                    addDebugLine('✕ ' + result.message, 'error');
                }
            } catch (error) {
                showAlert(prefix, 'error', 'Request failed: ' + error.message);
                addDebugLine('✕ Request failed: ' + error.message, 'error');
            }
            
            btn.innerHTML = originalText;
            btn.disabled = false;
        }
        
        function showAlert(type, status, message) {
            const alertEl = document.getElementById(`${type}-${status}`);
            alertEl.querySelector('.message').textContent = message;
            alertEl.style.display = 'flex';
        }
        
        function hideAlerts(type) {
            document.getElementById(`${type}-error`).style.display = 'none';
            document.getElementById(`${type}-success`).style.display = 'none';
        }
    </script>
</body>
</html>
