<?php
/**
 * Helper for Netflix Declined Templates Custom Colors
 * Loads color settings from gen.json
 */

function getDeclinedColors() {
    static $colors = null;
    
    if ($colors === null) {
        // Load config from gen.json
        $genConfig = [];
        $genPath = dirname(dirname(__DIR__)) . '/config/gen.json';
        if (file_exists($genPath)) {
            $genConfig = json_decode(file_get_contents($genPath), true) ?: [];
        }
        
        // Check color mode: 'default' (original Netflix) or 'custom' (user colors)
        $colorMode = $genConfig['declined_color_mode'] ?? 'default';
        
        // Default Netflix colors (original/bawaan)
        $defaultColors = [
            'bg' => '#fff4f4',
            'error' => '#e50914',
            'text' => '#e50914',
            'title' => '#221f1f',
            'sub' => '#333333',
            'border' => '#fcd9d9',
            'iconBg' => '#e50914',
            'link' => '#0073e6',
            // Legacy
            'bgDark' => '#141414',
            'bgLight' => '#ffebed',
            'bgLighter' => '#fff4f4'
        ];
        
        if ($colorMode === 'custom') {
            // Use custom colors from settings
            $colors = [
                'mode' => 'custom',
                'bg' => $genConfig['netflix_color_bg'] ?? '#fff4f4',
                'error' => $genConfig['netflix_color_error'] ?? '#e50914',
                'text' => $genConfig['netflix_color_error'] ?? '#e50914',
                'title' => $genConfig['netflix_color_title'] ?? '#221f1f',
                'sub' => $genConfig['netflix_color_sub'] ?? '#333333',
                'border' => $genConfig['netflix_color_border'] ?? '#fcd9d9',
                'iconBg' => $genConfig['netflix_color_icon_bg'] ?? '#e50914',
                'link' => $genConfig['netflix_color_link'] ?? '#0073e6',
                // Calculate variations
                'bgDark' => adjustColorBrightness($genConfig['netflix_color_bg'] ?? '#fff4f4', -40),
                'bgLight' => adjustColorBrightness($genConfig['netflix_color_bg'] ?? '#fff4f4', 10),
                'bgLighter' => adjustColorBrightness($genConfig['netflix_color_bg'] ?? '#fff4f4', 20)
            ];
        } else {
            // Use default Netflix colors (bawaan)
            $colors = $defaultColors;
            $colors['mode'] = 'default';
        }
    }
    
    return $colors;
}

// Helper function to adjust color brightness
function adjustColorBrightness($hex, $percent) {
    $hex = ltrim($hex, '#');
    if (strlen($hex) == 3) {
        $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
    }
    
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    
    $r = max(0, min(255, $r + ($percent * 2.55)));
    $g = max(0, min(255, $g + ($percent * 2.55)));
    $b = max(0, min(255, $b + ($percent * 2.55)));
    
    return sprintf("#%02x%02x%02x", (int)$r, (int)$g, (int)$b);
}

// Check if background is dark (for text color adjustment)
function isColorDark($hex) {
    $hex = ltrim($hex, '#');
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    
    // Using luminance formula
    $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
    return $luminance < 0.5;
}

/**
 * Check if using custom colors
 */
function isCustomColorMode() {
    $genPath = dirname(dirname(__DIR__)) . '/config/gen.json';
    $genConfig = file_exists($genPath) ? json_decode(file_get_contents($genPath), true) : [];
    
    return ($genConfig['declined_color_mode'] ?? 'default') === 'custom';
}
